"""
AI Job processor - Handles all AI job types
"""
from typing import Dict, Any
from datetime import datetime
from sqlalchemy import select

from app.database import get_db_session
from app.models import AIJob, CostTracking
from app.services.ai_providers import get_ai_provider
from app.utils.observability import logger, metrics_collector, cost_tracker
from app.celery_app import send_callback
from config import settings


async def process_job(job_id: str) -> Dict[str, Any]:
    """
    Process an AI job based on its type
    
    Main dispatcher for all AI job types
    """
    async with get_db_session() as db:
        # Get job
        stmt = select(AIJob).where(AIJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise ValueError(f"Job {job_id} not found")
        
        # Update status to processing
        job.status = "processing"
        job.started_at = datetime.utcnow()
        await db.commit()
        
        logger.info("Processing AI job", job_id=job_id, job_type=job.job_type)
        
        try:
            # Route to appropriate processor
            processors = {
                "proposal": process_proposal,
                "content_ideas": process_content_ideas,
                "auto_reply": process_auto_reply,
                "ads_generate": process_ads_generate,
                "forecast": process_forecast,
                "matchmaking": process_matchmaking,
                "translate": process_translate,
                "contract_draft": process_contract_draft,
                "support_rag": process_support_rag,
                "custom": process_custom,
            }
            
            processor = processors.get(job.job_type)
            if not processor:
                raise ValueError(f"Unknown job type: {job.job_type}")
            
            # Process job
            result = await processor(job.payload)
            
            # Update job with result
            job.status = "completed"
            job.result = result["content"]
            job.provider = result.get("provider")
            job.model_used = result.get("model")
            job.tokens_input = result.get("tokens_input", 0)
            job.tokens_output = result.get("tokens_output", 0)
            job.cost_usd = result.get("cost_usd", 0.0)
            job.completed_at = datetime.utcnow()
            
            # Track cost
            if settings.track_costs and job.cost_usd > 0:
                cost_record = CostTracking(
                    tenant_id=job.tenant_id,
                    job_id=job.id,
                    provider=job.provider,
                    model=job.model_used,
                    operation="completion",
                    tokens_input=job.tokens_input,
                    tokens_output=job.tokens_output,
                    tokens_total=job.tokens_input + job.tokens_output,
                    cost_usd=job.cost_usd
                )
                db.add(cost_record)
            
            await db.commit()
            
            # Track metrics
            duration = (job.completed_at - job.created_at).total_seconds()
            metrics_collector.track_job_completion(job.job_type, "completed", duration)
            
            logger.info(
                "AI job completed",
                job_id=job_id,
                job_type=job.job_type,
                cost_usd=job.cost_usd
            )
            
            # Send callback if configured
            if job.callback_url:
                await send_callback_async(job)
            
            return {"status": "completed", "job_id": job_id}
            
        except Exception as e:
            # Mark job as failed
            job.status = "failed"
            job.error_message = str(e)
            job.completed_at = datetime.utcnow()
            await db.commit()
            
            # Track metrics
            duration = (job.completed_at - job.created_at).total_seconds()
            metrics_collector.track_job_completion(job.job_type, "failed", duration)
            
            logger.error("AI job failed", job_id=job_id, error=str(e))
            
            # Send failure callback
            if job.callback_url:
                await send_callback_async(job)
            
            raise


async def process_proposal(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Generate business proposal"""
    provider = get_ai_provider()
    
    prompt = f"""Generate a professional business proposal based on the following details:

Client: {payload.get('client_name', 'N/A')}
Project: {payload.get('project_description', 'N/A')}
Budget: {payload.get('budget', 'N/A')}
Timeline: {payload.get('timeline', 'N/A')}
Additional Requirements: {payload.get('requirements', 'N/A')}

Create a comprehensive proposal including:
1. Executive Summary
2. Project Scope
3. Deliverables
4. Timeline
5. Budget Breakdown
6. Terms and Conditions
"""
    
    return await provider.generate_completion(prompt, max_tokens=2000)


async def process_content_ideas(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Generate content ideas"""
    provider = get_ai_provider()
    
    topic = payload.get('topic', '')
    count = payload.get('count', 5)
    tone = payload.get('tone', 'professional')
    platform = payload.get('platform', 'general')
    
    prompt = f"""Generate {count} creative content ideas for {platform} about: {topic}

Tone: {tone}
Target audience: {payload.get('target_audience', 'general audience')}

For each idea, provide:
- Title/Hook
- Brief description
- Key points to cover
- Suggested format (video, article, infographic, etc.)
"""
    
    return await provider.generate_completion(prompt, max_tokens=1500)


async def process_auto_reply(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Generate auto-reply to message"""
    provider = get_ai_provider()
    
    message = payload.get('message', '')
    context = payload.get('context', '')
    tone = payload.get('tone', 'friendly')
    
    prompt = f"""Generate a {tone} reply to the following message:

Message: {message}

Context: {context}

Guidelines:
- Be helpful and professional
- Address the main points
- Keep it concise
- Match the tone requested
"""
    
    return await provider.generate_completion(prompt, max_tokens=500, temperature=0.7)


async def process_ads_generate(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Generate ad variants"""
    provider = get_ai_provider()
    
    product = payload.get('product', '')
    target_audience = payload.get('target_audience', '')
    variants = payload.get('variants', 3)
    platform = payload.get('platform', 'social media')
    
    prompt = f"""Generate {variants} ad copy variants for {platform}:

Product/Service: {product}
Target Audience: {target_audience}
Key Benefits: {payload.get('benefits', 'N/A')}
Call to Action: {payload.get('cta', 'Learn More')}

For each variant, provide:
- Headline
- Body copy
- Call to action
- Suggested visual elements
"""
    
    return await provider.generate_completion(prompt, max_tokens=1500)


async def process_forecast(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Generate forecast/predictions"""
    provider = get_ai_provider()
    
    data = payload.get('historical_data', {})
    metric = payload.get('metric', 'sales')
    period = payload.get('forecast_period', '3 months')
    
    prompt = f"""Analyze the following data and provide a forecast for {metric} over the next {period}:

Historical Data: {data}

Provide:
1. Trend analysis
2. Forecast with confidence intervals
3. Key factors influencing the forecast
4. Recommendations
"""
    
    return await provider.generate_completion(prompt, max_tokens=1500)


async def process_matchmaking(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Match influencers with brands"""
    provider = get_ai_provider()
    
    brand_profile = payload.get('brand_profile', {})
    influencer_criteria = payload.get('criteria', {})
    
    prompt = f"""Analyze brand-influencer compatibility:

Brand Profile:
- Industry: {brand_profile.get('industry', 'N/A')}
- Target Audience: {brand_profile.get('target_audience', 'N/A')}
- Budget: {brand_profile.get('budget', 'N/A')}
- Campaign Goals: {brand_profile.get('goals', 'N/A')}

Influencer Criteria:
{influencer_criteria}

Provide:
1. Compatibility score (0-100)
2. Strengths of the match
3. Potential concerns
4. Recommendations for collaboration
"""
    
    return await provider.generate_completion(prompt, max_tokens=1000)


async def process_translate(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Translate content"""
    provider = get_ai_provider()
    
    text = payload.get('text', '')
    target_language = payload.get('target_language', 'Spanish')
    preserve_tone = payload.get('preserve_tone', True)
    
    prompt = f"""Translate the following text to {target_language}:

{text}

Requirements:
- Maintain the original meaning
- {'Preserve the tone and style' if preserve_tone else 'Adapt for target audience'}
- Use natural, native expressions
"""
    
    return await provider.generate_completion(prompt, max_tokens=2000)


async def process_contract_draft(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Draft contract"""
    provider = get_ai_provider()
    
    contract_type = payload.get('contract_type', 'service agreement')
    parties = payload.get('parties', {})
    terms = payload.get('terms', {})
    
    prompt = f"""Draft a {contract_type} with the following details:

Parties:
{parties}

Terms:
{terms}

Include standard clauses for:
- Scope of work
- Payment terms
- Confidentiality
- Termination
- Dispute resolution

Note: This is a draft and should be reviewed by legal counsel.
"""
    
    return await provider.generate_completion(prompt, max_tokens=2500)


async def process_support_rag(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Process support query using RAG"""
    from app.services.vector_store import vector_store_service
    
    query = payload.get('query', '')
    tenant_id = payload.get('tenant_id', '')
    
    # Search knowledge base
    results = await vector_store_service.search(
        tenant_id=tenant_id,
        query=query,
        top_k=3,
        document_type="support_doc"
    )
    
    if not results:
        provider = get_ai_provider()
        return await provider.generate_completion(
            f"Answer this support question: {query}",
            max_tokens=500
        )
    
    # Build context from results
    context = "\n\n".join([
        f"Source: {r['title']}\n{r['content']}"
        for r in results
    ])
    
    provider = get_ai_provider()
    prompt = f"""Based on the following knowledge base, answer the support question:

Knowledge Base:
{context}

Question: {query}

Provide a helpful, accurate answer based on the information above.
"""
    
    return await provider.generate_completion(prompt, max_tokens=800)


async def process_custom(payload: Dict[str, Any]) -> Dict[str, Any]:
    """Process custom job"""
    provider = get_ai_provider()
    
    prompt = payload.get('prompt', '')
    max_tokens = payload.get('max_tokens', 1000)
    temperature = payload.get('temperature', 0.7)
    
    return await provider.generate_completion(
        prompt,
        max_tokens=max_tokens,
        temperature=temperature
    )


async def send_callback_async(job: AIJob):
    """Send callback asynchronously"""
    from app.schemas import CallbackPayload
    
    payload = CallbackPayload(
        ai_job_id=job.id,
        status=job.status,
        result=job.result,
        error_message=job.error_message,
        provider=job.provider,
        model_used=job.model_used,
        tokens_input=job.tokens_input,
        tokens_output=job.tokens_output,
        cost_usd=job.cost_usd,
        completed_at=job.completed_at,
        metadata=job.metadata
    )
    
    # Enqueue callback task
    send_callback.apply_async(
        args=[job.callback_url, payload.dict()],
        retry=True
    )
    
    logger.info("Callback enqueued", job_id=job.id, callback_url=job.callback_url)