"""
RAG (Retrieval-Augmented Generation) router
Handles document ingestion and query endpoints
"""
from fastapi import APIRouter, Depends, HTTPException, status, Header
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select
from typing import Optional

from app.database import get_db
from app.models import VectorDocument
from app.schemas import (
    RAGIngestRequest, RAGIngestResponse,
    RAGQueryRequest, RAGQueryResponse
)
from app.utils.observability import logger
from config import settings

router = APIRouter()


async def verify_api_key(x_api_key: Optional[str] = Header(None)):
    """Verify API key from Laravel"""
    if not x_api_key or x_api_key != settings.laravel_api_key:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or missing API key"
        )
    return x_api_key


@router.post("/ingest", response_model=RAGIngestResponse)
async def ingest_documents(
    request: RAGIngestRequest,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Ingest documents into vector store
    
    Processes documents, generates embeddings, and stores in vector DB.
    """
    try:
        from app.services.vector_store import vector_store_service
        
        ingested_ids = []
        failed_count = 0
        
        for doc in request.documents:
            try:
                # Generate embedding and store
                vector_id = await vector_store_service.add_document(
                    tenant_id=request.tenant_id,
                    document_id=doc.get("id"),
                    content=doc.get("content"),
                    metadata={
                        "title": doc.get("title"),
                        "document_type": request.document_type,
                        **doc.get("metadata", {})
                    }
                )
                
                # Store in database
                vector_doc = VectorDocument(
                    tenant_id=request.tenant_id,
                    document_id=doc.get("id"),
                    document_type=request.document_type,
                    title=doc.get("title"),
                    content=doc.get("content"),
                    vector_id=vector_id,
                    embedding_model=settings.embedding_model,
                    metadata=doc.get("metadata"),
                    is_active=True
                )
                
                db.add(vector_doc)
                ingested_ids.append(doc.get("id"))
                
                logger.info(
                    "Document ingested",
                    document_id=doc.get("id"),
                    tenant_id=request.tenant_id
                )
                
            except Exception as e:
                logger.error(
                    "Failed to ingest document",
                    document_id=doc.get("id"),
                    error=str(e)
                )
                failed_count += 1
        
        await db.commit()
        
        return RAGIngestResponse(
            ingested_count=len(ingested_ids),
            failed_count=failed_count,
            document_ids=ingested_ids,
            message=f"Ingested {len(ingested_ids)} documents, {failed_count} failed"
        )
        
    except Exception as e:
        logger.error("Failed to ingest documents", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to ingest documents: {str(e)}"
        )


@router.post("/query", response_model=RAGQueryResponse)
async def query_rag(
    request: RAGQueryRequest,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Query RAG system
    
    Retrieves relevant documents and generates answer using LLM.
    """
    try:
        from app.services.vector_store import vector_store_service
        from app.services.ai_providers import get_ai_provider
        
        # Search vector store
        search_results = await vector_store_service.search(
            tenant_id=request.tenant_id,
            query=request.query,
            top_k=request.top_k,
            document_type=request.document_type
        )
        
        if not search_results:
            return RAGQueryResponse(
                answer="I couldn't find any relevant information to answer your question.",
                sources=[] if request.include_sources else None,
                confidence=0.0,
                tokens_used=0,
                cost_usd=0.0
            )
        
        # Build context from search results
        context = "\n\n".join([
            f"Document: {result['title']}\n{result['content']}"
            for result in search_results
        ])
        
        # Generate answer using LLM
        provider = get_ai_provider("openai")  # Default to OpenAI
        
        prompt = f"""Based on the following context, answer the user's question.
If the context doesn't contain relevant information, say so.

Context:
{context}

Question: {request.query}

Answer:"""
        
        response = await provider.generate_completion(
            prompt=prompt,
            max_tokens=500,
            temperature=0.3
        )
        
        # Prepare sources if requested
        sources = None
        if request.include_sources:
            sources = [
                {
                    "document_id": result["document_id"],
                    "title": result["title"],
                    "excerpt": result["content"][:200] + "...",
                    "score": result.get("score", 0.0)
                }
                for result in search_results
            ]
        
        logger.info(
            "RAG query completed",
            tenant_id=request.tenant_id,
            query_length=len(request.query),
            results_count=len(search_results)
        )
        
        return RAGQueryResponse(
            answer=response["content"],
            sources=sources,
            confidence=search_results[0].get("score", 0.0) if search_results else 0.0,
            tokens_used=response.get("tokens_used", 0),
            cost_usd=response.get("cost_usd", 0.0)
        )
        
    except Exception as e:
        logger.error("Failed to query RAG", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to query RAG: {str(e)}"
        )


@router.delete("/documents/{document_id}")
async def delete_document(
    document_id: str,
    tenant_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Delete a document from vector store
    
    Marks document as inactive and removes from vector store.
    """
    try:
        stmt = select(VectorDocument).where(
            VectorDocument.document_id == document_id,
            VectorDocument.tenant_id == tenant_id
        )
        result = await db.execute(stmt)
        doc = result.scalar_one_or_none()
        
        if not doc:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Document {document_id} not found"
            )
        
        # Remove from vector store
        from app.services.vector_store import vector_store_service
        await vector_store_service.delete_document(doc.vector_id)
        
        # Mark as inactive
        doc.is_active = False
        await db.commit()
        
        logger.info(
            "Document deleted",
            document_id=document_id,
            tenant_id=tenant_id
        )
        
        return {"message": "Document deleted successfully", "document_id": document_id}
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to delete document", document_id=document_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to delete document: {str(e)}"
        )


@router.get("/documents")
async def list_documents(
    tenant_id: str,
    document_type: Optional[str] = None,
    limit: int = 50,
    offset: int = 0,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    List documents for a tenant
    
    Returns paginated list of documents.
    """
    try:
        stmt = select(VectorDocument).where(
            VectorDocument.tenant_id == tenant_id,
            VectorDocument.is_active == True
        )
        
        if document_type:
            stmt = stmt.where(VectorDocument.document_type == document_type)
        
        stmt = stmt.order_by(VectorDocument.created_at.desc())
        stmt = stmt.limit(limit).offset(offset)
        
        result = await db.execute(stmt)
        documents = result.scalars().all()
        
        return {
            "documents": [
                {
                    "document_id": doc.document_id,
                    "title": doc.title,
                    "document_type": doc.document_type,
                    "created_at": doc.created_at,
                    "metadata": doc.metadata
                }
                for doc in documents
            ],
            "total": len(documents),
            "limit": limit,
            "offset": offset
        }
        
    except Exception as e:
        logger.error("Failed to list documents", tenant_id=tenant_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to list documents: {str(e)}"
        )