"""
Social Media Publishers router
Handles publishing content to various social platforms
"""
from fastapi import APIRouter, Depends, HTTPException, status, Header
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select
from typing import Optional
from datetime import datetime

from app.database import get_db
from app.models import PublishJob
from app.schemas import PublishRequest, PublishResponse, PublishStatus
from app.utils.observability import logger, metrics_collector
from config import settings

router = APIRouter()


async def verify_api_key(x_api_key: Optional[str] = Header(None)):
    """Verify API key from Laravel"""
    if not x_api_key or x_api_key != settings.laravel_api_key:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or missing API key"
        )
    return x_api_key


@router.post("", response_model=PublishResponse)
async def publish_content(
    request: PublishRequest,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Publish content to social media platform
    
    Supports idempotency and scheduled publishing.
    Returns immediately with job ID, actual publishing happens async.
    """
    try:
        # Check for idempotency
        if request.idempotency_key:
            stmt = select(PublishJob).where(
                PublishJob.idempotency_key == request.idempotency_key
            )
            result = await db.execute(stmt)
            existing_job = result.scalar_one_or_none()
            
            if existing_job:
                logger.info(
                    "Returning existing publish job",
                    job_id=existing_job.id,
                    platform=existing_job.platform
                )
                return PublishResponse(
                    publish_job_id=existing_job.id,
                    status=PublishStatus(existing_job.status),
                    platform=existing_job.platform,
                    message="Publish job already exists (idempotent)",
                    platform_post_id=existing_job.platform_post_id
                )
        
        # Create publish job
        job = PublishJob(
            tenant_id=request.tenant_id,
            owner_id=request.owner_id,
            platform=request.platform.value,
            content=request.content,
            callback_url=request.callback_url,
            idempotency_key=request.idempotency_key,
            status="pending"
        )
        
        db.add(job)
        await db.commit()
        await db.refresh(job)
        
        logger.info(
            "Publish job created",
            job_id=job.id,
            platform=job.platform,
            tenant_id=job.tenant_id
        )
        
        # Enqueue to Celery
        from app.celery_app import process_publish_job
        
        if request.schedule_at:
            # Schedule for future
            task = process_publish_job.apply_async(
                args=[job.id],
                eta=request.schedule_at
            )
            logger.info(
                "Publish job scheduled",
                job_id=job.id,
                scheduled_at=request.schedule_at
            )
        else:
            # Publish immediately
            task = process_publish_job.apply_async(args=[job.id])
            logger.info("Publish job enqueued", job_id=job.id)
        
        return PublishResponse(
            publish_job_id=job.id,
            status=PublishStatus.PENDING,
            platform=job.platform,
            message="Publish job queued successfully"
        )
        
    except Exception as e:
        logger.error(
            "Failed to create publish job",
            error=str(e),
            platform=request.platform.value
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to create publish job: {str(e)}"
        )


@router.get("/{job_id}")
async def get_publish_status(
    job_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Get status of a publish job
    
    Returns job details including platform post ID if published.
    """
    try:
        stmt = select(PublishJob).where(PublishJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Publish job {job_id} not found"
            )
        
        return {
            "publish_job_id": job.id,
            "platform": job.platform,
            "status": job.status,
            "platform_post_id": job.platform_post_id,
            "platform_response": job.platform_response,
            "error_message": job.error_message,
            "retry_count": job.retry_count,
            "created_at": job.created_at,
            "published_at": job.published_at,
            "metadata": job.metadata
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to get publish status", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to get publish status: {str(e)}"
        )


@router.delete("/{job_id}")
async def cancel_publish(
    job_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Cancel a pending publish job
    
    Only pending jobs can be cancelled.
    """
    try:
        stmt = select(PublishJob).where(PublishJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Publish job {job_id} not found"
            )
        
        if job.status != "pending":
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Cannot cancel job in status: {job.status}"
            )
        
        job.status = "failed"
        job.error_message = "Cancelled by user"
        await db.commit()
        
        logger.info("Publish job cancelled", job_id=job_id)
        
        return {"message": "Publish job cancelled", "job_id": job_id}
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to cancel publish job", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to cancel publish job: {str(e)}"
        )


@router.post("/{job_id}/retry")
async def retry_publish(
    job_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Retry a failed publish job
    
    Resets status to pending and re-enqueues.
    """
    try:
        stmt = select(PublishJob).where(PublishJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Publish job {job_id} not found"
            )
        
        if job.status not in ["failed"]:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Cannot retry job in status: {job.status}"
            )
        
        if job.retry_count >= job.max_retries:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Maximum retry attempts reached"
            )
        
        job.status = "pending"
        job.retry_count += 1
        job.error_message = None
        await db.commit()
        
        # Re-enqueue
        from app.celery_app import process_publish_job
        task = process_publish_job.apply_async(args=[job.id])
        
        logger.info("Publish job retried", job_id=job_id, retry_count=job.retry_count)
        
        return {
            "message": "Publish job retried",
            "job_id": job_id,
            "retry_count": job.retry_count
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to retry publish job", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to retry publish job: {str(e)}"
        )