"""
Database models for AI Microservice
SQLAlchemy models for job tracking, cost tracking, and audit logs
"""
from datetime import datetime
from typing import Optional, Dict, Any
from sqlalchemy import Column, String, Integer, Float, DateTime, JSON, Text, Boolean, Index
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.sql import func
import uuid

Base = declarative_base()


def generate_uuid():
    """Generate UUID for primary keys"""
    return str(uuid.uuid4())


class AIJob(Base):
    """AI Job tracking model"""
    __tablename__ = "ai_jobs"
    
    id = Column(String(36), primary_key=True, default=generate_uuid)
    job_type = Column(String(50), nullable=False, index=True)
    tenant_id = Column(String(36), nullable=False, index=True)
    owner_id = Column(String(36), nullable=False, index=True)
    
    # Job status: pending, processing, completed, failed, cancelled
    status = Column(String(20), nullable=False, default="pending", index=True)
    
    # Idempotency
    idempotency_key = Column(String(255), unique=True, nullable=True, index=True)
    
    # Job data
    payload = Column(JSON, nullable=False)
    result = Column(JSON, nullable=True)
    error_message = Column(Text, nullable=True)
    
    # Callback
    callback_url = Column(String(500), nullable=True)
    callback_status = Column(String(20), nullable=True)  # pending, sent, failed
    callback_attempts = Column(Integer, default=0)
    callback_last_attempt = Column(DateTime, nullable=True)
    
    # Provider info
    provider = Column(String(50), nullable=True)  # openai, anthropic, local
    model_used = Column(String(100), nullable=True)
    
    # Cost tracking
    tokens_input = Column(Integer, default=0)
    tokens_output = Column(Integer, default=0)
    cost_usd = Column(Float, default=0.0)
    
    # Timing
    created_at = Column(DateTime, nullable=False, default=func.now(), index=True)
    started_at = Column(DateTime, nullable=True)
    completed_at = Column(DateTime, nullable=True)
    
    # Metadata
    metadata = Column(JSON, nullable=True)
    
    __table_args__ = (
        Index('idx_tenant_status', 'tenant_id', 'status'),
        Index('idx_owner_created', 'owner_id', 'created_at'),
    )
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary"""
        return {
            "id": self.id,
            "job_type": self.job_type,
            "tenant_id": self.tenant_id,
            "owner_id": self.owner_id,
            "status": self.status,
            "payload": self.payload,
            "result": self.result,
            "error_message": self.error_message,
            "callback_url": self.callback_url,
            "callback_status": self.callback_status,
            "provider": self.provider,
            "model_used": self.model_used,
            "tokens_input": self.tokens_input,
            "tokens_output": self.tokens_output,
            "cost_usd": self.cost_usd,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "started_at": self.started_at.isoformat() if self.started_at else None,
            "completed_at": self.completed_at.isoformat() if self.completed_at else None,
            "metadata": self.metadata,
        }


class PublishJob(Base):
    """Social media publish job tracking"""
    __tablename__ = "publish_jobs"
    
    id = Column(String(36), primary_key=True, default=generate_uuid)
    tenant_id = Column(String(36), nullable=False, index=True)
    owner_id = Column(String(36), nullable=False, index=True)
    
    # Platform: twitter, facebook, instagram, youtube, linkedin
    platform = Column(String(50), nullable=False, index=True)
    
    # Status: pending, publishing, published, failed
    status = Column(String(20), nullable=False, default="pending", index=True)
    
    # Idempotency
    idempotency_key = Column(String(255), unique=True, nullable=True, index=True)
    
    # Content
    content = Column(JSON, nullable=False)  # text, media_urls, etc.
    
    # Platform response
    platform_post_id = Column(String(255), nullable=True)
    platform_response = Column(JSON, nullable=True)
    error_message = Column(Text, nullable=True)
    
    # Retry logic
    retry_count = Column(Integer, default=0)
    max_retries = Column(Integer, default=3)
    next_retry_at = Column(DateTime, nullable=True)
    
    # Callback
    callback_url = Column(String(500), nullable=True)
    callback_status = Column(String(20), nullable=True)
    
    # Timing
    created_at = Column(DateTime, nullable=False, default=func.now(), index=True)
    published_at = Column(DateTime, nullable=True)
    
    # Metadata
    metadata = Column(JSON, nullable=True)
    
    __table_args__ = (
        Index('idx_tenant_platform', 'tenant_id', 'platform'),
        Index('idx_status_retry', 'status', 'next_retry_at'),
    )
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary"""
        return {
            "id": self.id,
            "tenant_id": self.tenant_id,
            "owner_id": self.owner_id,
            "platform": self.platform,
            "status": self.status,
            "content": self.content,
            "platform_post_id": self.platform_post_id,
            "platform_response": self.platform_response,
            "error_message": self.error_message,
            "retry_count": self.retry_count,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "published_at": self.published_at.isoformat() if self.published_at else None,
            "metadata": self.metadata,
        }


class CostTracking(Base):
    """Cost tracking for AI operations"""
    __tablename__ = "cost_tracking"
    
    id = Column(String(36), primary_key=True, default=generate_uuid)
    tenant_id = Column(String(36), nullable=False, index=True)
    job_id = Column(String(36), nullable=True, index=True)
    
    # Provider and model
    provider = Column(String(50), nullable=False, index=True)
    model = Column(String(100), nullable=False)
    operation = Column(String(50), nullable=False)  # completion, embedding, etc.
    
    # Usage
    tokens_input = Column(Integer, default=0)
    tokens_output = Column(Integer, default=0)
    tokens_total = Column(Integer, default=0)
    
    # Cost
    cost_usd = Column(Float, nullable=False)
    cost_currency = Column(String(3), default="USD")
    
    # Timing
    created_at = Column(DateTime, nullable=False, default=func.now(), index=True)
    
    # Metadata
    metadata = Column(JSON, nullable=True)
    
    __table_args__ = (
        Index('idx_tenant_created', 'tenant_id', 'created_at'),
        Index('idx_provider_created', 'provider', 'created_at'),
    )


class VectorDocument(Base):
    """Vector document storage for RAG"""
    __tablename__ = "vector_documents"
    
    id = Column(String(36), primary_key=True, default=generate_uuid)
    tenant_id = Column(String(36), nullable=False, index=True)
    
    # Document info
    document_id = Column(String(255), nullable=False, index=True)
    document_type = Column(String(50), nullable=False)  # support_doc, knowledge_base, etc.
    title = Column(String(500), nullable=True)
    content = Column(Text, nullable=False)
    
    # Vector info
    vector_id = Column(String(255), nullable=True)  # ID in vector store
    embedding_model = Column(String(100), nullable=False)
    
    # Metadata
    metadata = Column(JSON, nullable=True)
    tags = Column(JSON, nullable=True)
    
    # Status
    is_active = Column(Boolean, default=True, index=True)
    
    # Timing
    created_at = Column(DateTime, nullable=False, default=func.now(), index=True)
    updated_at = Column(DateTime, nullable=False, default=func.now(), onupdate=func.now())
    
    __table_args__ = (
        Index('idx_tenant_type', 'tenant_id', 'document_type'),
        Index('idx_tenant_active', 'tenant_id', 'is_active'),
    )


class AuditLog(Base):
    """Audit log for security and compliance"""
    __tablename__ = "audit_logs"
    
    id = Column(String(36), primary_key=True, default=generate_uuid)
    tenant_id = Column(String(36), nullable=True, index=True)
    user_id = Column(String(36), nullable=True, index=True)
    
    # Action info
    action = Column(String(100), nullable=False, index=True)
    resource_type = Column(String(50), nullable=False)
    resource_id = Column(String(36), nullable=True)
    
    # Details
    details = Column(JSON, nullable=True)
    ip_address = Column(String(45), nullable=True)
    user_agent = Column(String(500), nullable=True)
    
    # Result
    status = Column(String(20), nullable=False)  # success, failure
    error_message = Column(Text, nullable=True)
    
    # Timing
    created_at = Column(DateTime, nullable=False, default=func.now(), index=True)
    
    __table_args__ = (
        Index('idx_tenant_action', 'tenant_id', 'action'),
        Index('idx_user_created', 'user_id', 'created_at'),
    )