"""
Celery application configuration and tasks
"""
from celery import Celery
from celery.signals import task_prerun, task_postrun, task_failure
from datetime import datetime
import asyncio

from config import settings
from app.utils.observability import logger, metrics_collector

# Create Celery app
celery_app = Celery(
    "ai_microservice",
    broker=settings.celery_broker_url,
    backend=settings.celery_result_backend,
    include=[
        "app.celery_app"
    ]
)

# Configure Celery
celery_app.conf.update(
    task_serializer="json",
    accept_content=["json"],
    result_serializer="json",
    timezone="UTC",
    enable_utc=True,
    task_track_started=settings.celery_task_track_started,
    task_time_limit=settings.celery_task_time_limit,
    worker_prefetch_multiplier=settings.celery_worker_prefetch_multiplier,
    task_acks_late=True,
    task_reject_on_worker_lost=True,
    task_default_priority=5,
    task_routes={
        "app.celery_app.process_ai_job": {"queue": "ai_jobs"},
        "app.celery_app.process_publish_job": {"queue": "publishers"},
        "app.celery_app.send_callback": {"queue": "callbacks"},
    }
)


# Celery signals for monitoring
@task_prerun.connect
def task_prerun_handler(task_id, task, *args, **kwargs):
    """Log task start"""
    logger.info(
        "Task started",
        task_id=task_id,
        task_name=task.name
    )


@task_postrun.connect
def task_postrun_handler(task_id, task, *args, **kwargs):
    """Log task completion"""
    logger.info(
        "Task completed",
        task_id=task_id,
        task_name=task.name
    )


@task_failure.connect
def task_failure_handler(task_id, exception, *args, **kwargs):
    """Log task failure"""
    logger.error(
        "Task failed",
        task_id=task_id,
        error=str(exception),
        error_type=type(exception).__name__
    )


def run_async(coro):
    """Helper to run async functions in Celery tasks"""
    loop = asyncio.get_event_loop()
    return loop.run_until_complete(coro)


@celery_app.task(bind=True, max_retries=3)
def process_ai_job(self, job_id: str):
    """
    Process an AI job
    
    Main task that handles all AI job types.
    """
    from app.services.job_processor import process_job
    
    try:
        logger.info("Processing AI job", job_id=job_id, task_id=self.request.id)
        
        # Run async job processor
        result = run_async(process_job(job_id))
        
        logger.info("AI job processed successfully", job_id=job_id)
        return result
        
    except Exception as e:
        logger.error(
            "AI job processing failed",
            job_id=job_id,
            error=str(e),
            retry_count=self.request.retries
        )
        
        # Retry with exponential backoff
        if self.request.retries < self.max_retries:
            raise self.retry(exc=e, countdown=2 ** self.request.retries)
        
        # Mark job as failed after max retries
        run_async(mark_job_failed(job_id, str(e)))
        raise


@celery_app.task(bind=True, max_retries=3)
def process_publish_job(self, job_id: str):
    """
    Process a social media publish job
    
    Handles publishing to various social platforms.
    """
    from app.services.publisher_processor import process_publish
    
    try:
        logger.info("Processing publish job", job_id=job_id, task_id=self.request.id)
        
        # Run async publisher
        result = run_async(process_publish(job_id))
        
        logger.info("Publish job processed successfully", job_id=job_id)
        return result
        
    except Exception as e:
        logger.error(
            "Publish job processing failed",
            job_id=job_id,
            error=str(e),
            retry_count=self.request.retries
        )
        
        # Retry with exponential backoff
        if self.request.retries < self.max_retries:
            raise self.retry(exc=e, countdown=2 ** self.request.retries)
        
        # Mark job as failed after max retries
        run_async(mark_publish_failed(job_id, str(e)))
        raise


@celery_app.task(bind=True, max_retries=5)
def send_callback(self, callback_url: str, payload: dict, signature: str = None):
    """
    Send callback to Laravel
    
    Handles callback delivery with retries.
    """
    import httpx
    from app.utils.security import SignatureValidator
    
    try:
        logger.info("Sending callback", callback_url=callback_url)
        
        headers = {
            "Content-Type": "application/json",
            "Authorization": f"Bearer {settings.laravel_api_key}"
        }
        
        # Add signature if provided
        if signature:
            headers["X-Signature"] = signature
        else:
            # Generate signature
            import json
            payload_str = json.dumps(payload, sort_keys=True)
            headers["X-Signature"] = SignatureValidator.generate_signature(
                payload_str,
                settings.laravel_callback_secret
            )
        
        # Send callback synchronously
        with httpx.Client(timeout=30.0) as client:
            response = client.post(
                callback_url,
                json=payload,
                headers=headers
            )
            response.raise_for_status()
        
        logger.info("Callback sent successfully", callback_url=callback_url)
        return {"status": "success", "status_code": response.status_code}
        
    except httpx.HTTPStatusError as e:
        logger.error(
            "Callback HTTP error",
            callback_url=callback_url,
            status_code=e.response.status_code,
            error=e.response.text
        )
        
        # Retry on 5xx errors
        if 500 <= e.response.status_code < 600:
            if self.request.retries < self.max_retries:
                raise self.retry(exc=e, countdown=2 ** self.request.retries)
        
        raise
        
    except Exception as e:
        logger.error(
            "Callback failed",
            callback_url=callback_url,
            error=str(e)
        )
        
        # Retry
        if self.request.retries < self.max_retries:
            raise self.retry(exc=e, countdown=2 ** self.request.retries)
        
        raise


async def mark_job_failed(job_id: str, error_message: str):
    """Mark AI job as failed"""
    from app.database import get_db_session
    from app.models import AIJob
    from sqlalchemy import select
    
    async with get_db_session() as db:
        stmt = select(AIJob).where(AIJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if job:
            job.status = "failed"
            job.error_message = error_message
            job.completed_at = datetime.utcnow()
            await db.commit()
            
            # Track metrics
            duration = (job.completed_at - job.created_at).total_seconds()
            metrics_collector.track_job_completion(job.job_type, "failed", duration)


async def mark_publish_failed(job_id: str, error_message: str):
    """Mark publish job as failed"""
    from app.database import get_db_session
    from app.models import PublishJob
    from sqlalchemy import select
    
    async with get_db_session() as db:
        stmt = select(PublishJob).where(PublishJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if job:
            job.status = "failed"
            job.error_message = error_message
            await db.commit()


# Periodic tasks
@celery_app.task
def cleanup_old_jobs():
    """
    Cleanup old completed jobs
    
    Runs periodically to remove old job records.
    """
    from datetime import timedelta
    
    logger.info("Running job cleanup task")
    
    # This would delete jobs older than 30 days
    # Implementation depends on requirements
    
    return {"status": "completed"}


@celery_app.task
def update_metrics():
    """
    Update system metrics
    
    Runs periodically to update Prometheus metrics.
    """
    try:
        # Update worker count
        inspect = celery_app.control.inspect()
        stats = inspect.stats()
        worker_count = len(stats) if stats else 0
        metrics_collector.update_worker_count(worker_count)
        
        # Update queue lengths
        # This requires additional Redis queries
        
        return {"status": "completed", "workers": worker_count}
        
    except Exception as e:
        logger.error("Failed to update metrics", error=str(e))
        return {"status": "failed", "error": str(e)}


# Configure periodic tasks
celery_app.conf.beat_schedule = {
    "cleanup-old-jobs": {
        "task": "app.celery_app.cleanup_old_jobs",
        "schedule": 3600.0,  # Every hour
    },
    "update-metrics": {
        "task": "app.celery_app.update_metrics",
        "schedule": 60.0,  # Every minute
    },
}