<?php

namespace Tests\Feature;

use App\Models\Brand;
use App\Models\User;
use App\Models\LocalizedContent;
use App\Models\LocalizationCache;
use App\Services\LocalizationService;
use App\Services\AITranslationService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use Mockery;

class LocalizationTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected User $brandAdmin;
    protected Brand $brand;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a brand
        $this->brand = Brand::factory()->create([
            'name' => 'Test Brand',
        ]);

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'role' => 'brand_admin',
            'brand_id' => $this->brand->id,
        ]);
    }

    /** @test */
    public function brand_admin_can_access_localization_index()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.localization.index', ['brand_id' => $this->brand->id]));

        $response->assertStatus(200);
        $response->assertViewIs('brand.localization.index');
        $response->assertViewHas(['brand', 'stats', 'supportedLocales', 'supportedTones']);
    }

    /** @test */
    public function brand_admin_can_request_localization()
    {
        // Mock the AI translation service
        $this->mock(AITranslationService::class, function ($mock) {
            $mock->shouldReceive('translate')
                ->once()
                ->andReturn([
                    'translated_text' => 'Texto traducido',
                    'cultural_notes' => 'Spanish translation notes',
                    'job_id' => 'test_job_123',
                ]);
        });

        $data = [
            'brand_id' => $this->brand->id,
            'content_type' => 'post',
            'content_id' => 1,
            'source_text' => 'Hello, this is a test post',
            'target_locales' => ['es-ES', 'fr-FR'],
            'tone' => 'friendly',
            'context' => 'Social media post for product launch',
        ];

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.localization.store'), $data);

        $response->assertStatus(201);
        $response->assertJson([
            'success' => true,
            'message' => 'Localization request processed successfully',
        ]);

        // Verify database records
        $this->assertDatabaseHas('localized_contents', [
            'brand_id' => $this->brand->id,
            'content_type' => 'post',
            'content_id' => 1,
            'locale_code' => 'es-ES',
            'status' => 'generated',
        ]);
    }

    /** @test */
    public function localization_request_validates_required_fields()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.localization.store'), []);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors([
            'brand_id',
            'content_type',
            'content_id',
            'source_text',
            'target_locales',
            'tone',
        ]);
    }

    /** @test */
    public function brand_admin_can_approve_localization()
    {
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'generated',
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.localization.approve', $localization));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Localization approved successfully',
        ]);

        $this->assertDatabaseHas('localized_contents', [
            'id' => $localization->id,
            'status' => 'approved',
            'approved_by' => $this->brandAdmin->id,
        ]);

        // Verify audit log
        $this->assertDatabaseHas('localization_audit_logs', [
            'localized_content_id' => $localization->id,
            'user_id' => $this->brandAdmin->id,
            'action' => 'approved',
        ]);
    }

    /** @test */
    public function brand_admin_can_reject_localization()
    {
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'generated',
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.localization.reject', $localization), [
                'reason' => 'Translation quality is not acceptable',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Localization rejected',
        ]);

        $this->assertDatabaseHas('localized_contents', [
            'id' => $localization->id,
            'status' => 'rejected',
        ]);

        // Verify audit log
        $this->assertDatabaseHas('localization_audit_logs', [
            'localized_content_id' => $localization->id,
            'user_id' => $this->brandAdmin->id,
            'action' => 'rejected',
        ]);
    }

    /** @test */
    public function brand_admin_can_edit_localization()
    {
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $this->brand->id,
            'localized_text' => 'Original translation',
            'created_by' => $this->brandAdmin->id,
        ]);

        $newText = 'Updated translation text';

        $response = $this->actingAs($this->brandAdmin)
            ->putJson(route('brand.localization.update', $localization), [
                'localized_text' => $newText,
                'notes' => 'Manual correction for better accuracy',
            ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Localization updated successfully',
        ]);

        $this->assertDatabaseHas('localized_contents', [
            'id' => $localization->id,
            'localized_text' => $newText,
            'status' => 'edited',
        ]);

        // Verify audit log
        $this->assertDatabaseHas('localization_audit_logs', [
            'localized_content_id' => $localization->id,
            'user_id' => $this->brandAdmin->id,
            'action' => 'edited',
        ]);
    }

    /** @test */
    public function brand_admin_can_export_localization()
    {
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $this->brand->id,
            'locale_code' => 'es-ES',
            'source_text' => 'Hello World',
            'localized_text' => 'Hola Mundo',
            'cultural_notes' => 'Spanish greeting',
            'tone' => 'friendly',
            'status' => 'approved',
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->getJson(route('brand.localization.export', $localization));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);

        $response->assertJsonStructure([
            'success',
            'data' => [
                'locale_code',
                'locale_name',
                'source_text',
                'localized_text',
                'cultural_notes',
                'tone',
                'status',
                'created_at',
            ],
        ]);

        // Verify audit log
        $this->assertDatabaseHas('localization_audit_logs', [
            'localized_content_id' => $localization->id,
            'user_id' => $this->brandAdmin->id,
            'action' => 'exported',
        ]);
    }

    /** @test */
    public function brand_admin_can_view_localization_stats()
    {
        // Create various localizations
        LocalizedContent::factory()->count(5)->create([
            'brand_id' => $this->brand->id,
            'status' => 'approved',
        ]);

        LocalizedContent::factory()->count(3)->create([
            'brand_id' => $this->brand->id,
            'status' => 'generated',
        ]);

        LocalizedContent::factory()->count(2)->create([
            'brand_id' => $this->brand->id,
            'status' => 'rejected',
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->getJson(route('brand.localization.stats', $this->brand));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'data' => [
                'total' => 10,
                'approved' => 5,
                'pending' => 3,
                'rejected' => 2,
            ],
        ]);
    }

    /** @test */
    public function localization_uses_cache_for_duplicate_requests()
    {
        $sourceText = 'Hello, this is a test';
        $translatedText = 'Hola, esto es una prueba';

        // Create a cache entry
        LocalizationCache::create([
            'source_text_hash' => LocalizationCache::generateHash($sourceText, 'en', 'es-ES', 'neutral'),
            'source_locale' => 'en',
            'target_locale' => 'es-ES',
            'tone' => 'neutral',
            'source_text' => $sourceText,
            'translated_text' => $translatedText,
            'cultural_notes' => 'Cached translation',
            'ai_provider' => 'openai',
            'usage_count' => 1,
            'last_used_at' => now(),
        ]);

        // Mock should not be called since we're using cache
        $this->mock(AITranslationService::class, function ($mock) {
            $mock->shouldNotReceive('translate');
        });

        $data = [
            'brand_id' => $this->brand->id,
            'content_type' => 'post',
            'content_id' => 1,
            'source_text' => $sourceText,
            'target_locales' => ['es-ES'],
            'tone' => 'neutral',
        ];

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.localization.store'), $data);

        $response->assertStatus(201);

        // Verify cache was used
        $this->assertDatabaseHas('localization_cache', [
            'source_text' => $sourceText,
            'target_locale' => 'es-ES',
            'usage_count' => 2, // Incremented
        ]);
    }

    /** @test */
    public function unauthorized_user_cannot_access_localization()
    {
        $otherUser = User::factory()->create([
            'role' => 'creator',
        ]);

        $response = $this->actingAs($otherUser)
            ->get(route('brand.localization.index', ['brand_id' => $this->brand->id]));

        $response->assertStatus(403);
    }

    /** @test */
    public function brand_admin_cannot_access_other_brands_localizations()
    {
        $otherBrand = Brand::factory()->create();
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $otherBrand->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->getJson(route('brand.localization.show', $localization));

        $response->assertStatus(403);
    }

    /** @test */
    public function can_get_supported_locales()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->getJson(route('brand.localization.supported-locales'));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);

        $response->assertJsonStructure([
            'success',
            'data',
        ]);
    }

    /** @test */
    public function can_get_supported_tones()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->getJson(route('brand.localization.supported-tones'));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);

        $response->assertJsonStructure([
            'success',
            'data',
        ]);
    }

    /** @test */
    public function can_delete_localization()
    {
        $localization = LocalizedContent::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->deleteJson(route('brand.localization.destroy', $localization));

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Localization deleted successfully',
        ]);

        $this->assertSoftDeleted('localized_contents', [
            'id' => $localization->id,
        ]);
    }
}