<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Campaign;
use App\Models\CampaignFinancial;
use App\Models\FinancialReport;
use App\Models\BudgetScenario;
use App\Services\FinancialInsightsService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Carbon\Carbon;

class FinancialInsightsTest extends TestCase
{
    use RefreshDatabase;

    private User $brand;
    private Campaign $campaign;
    private FinancialInsightsService $service;

    protected function setUp(): void
    {
        parent::setUp();

        $this->brand = User::factory()->brand()->create();
        $this->campaign = Campaign::factory()->create(['brand_id' => $this->brand->id]);
        $this->service = app(FinancialInsightsService::class);
    }

    /** @test */
    public function brand_can_view_financial_dashboard()
    {
        $this->actingAs($this->brand);

        $response = $this->get(route('brand.financial-insights.index'));

        $response->assertStatus(200);
        $response->assertViewIs('brand.financial-insights.index');
        $response->assertViewHas('data');
    }

    /** @test */
    public function non_brand_user_cannot_access_financial_insights()
    {
        $creator = User::factory()->creator()->create();
        $this->actingAs($creator);

        $response = $this->get(route('brand.financial-insights.index'));

        $response->assertStatus(403);
    }

    /** @test */
    public function can_update_campaign_financial_data()
    {
        $this->actingAs($this->brand);

        $data = [
            'spend_data' => [
                'ad_spend' => 5000,
                'influencer_payments' => 3000,
                'production_costs' => 1000,
                'other_costs' => 500,
            ],
            'performance_data' => [
                'clicks' => 10000,
                'impressions' => 50000,
                'conversions' => 200,
                'avg_order_value' => 75,
            ],
            'period_start' => '2024-01-01',
            'period_end' => '2024-01-31',
        ];

        $response = $this->postJson(
            route('brand.financial-insights.update-financials', $this->campaign),
            $data
        );

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $this->assertDatabaseHas('campaign_financials', [
            'campaign_id' => $this->campaign->id,
            'ad_spend' => 5000,
            'conversions' => 200,
        ]);
    }

    /** @test */
    public function financial_metrics_are_calculated_automatically()
    {
        $financial = CampaignFinancial::factory()->create([
            'campaign_id' => $this->campaign->id,
            'brand_id' => $this->brand->id,
            'ad_spend' => 5000,
            'influencer_payments' => 3000,
            'production_costs' => 1000,
            'other_costs' => 500,
            'clicks' => 10000,
            'conversions' => 200,
            'avg_order_value' => 75,
        ]);

        $financial->refresh();

        // Total spend should be sum of all costs
        $this->assertEquals(9500, $financial->total_spend);

        // Total revenue should be conversions * avg_order_value
        $this->assertEquals(15000, $financial->total_revenue);

        // ROI should be calculated
        $this->assertNotNull($financial->roi_percentage);
        $this->assertGreaterThan(0, $financial->roi_percentage);

        // Cost per click should be calculated
        $this->assertEquals(0.95, $financial->cost_per_click);

        // Cost per conversion should be calculated
        $this->assertEquals(47.5, $financial->cost_per_conversion);
    }

    /** @test */
    public function can_generate_campaign_financial_report()
    {
        CampaignFinancial::factory()->count(3)->create([
            'campaign_id' => $this->campaign->id,
            'brand_id' => $this->brand->id,
        ]);

        $this->actingAs($this->brand);

        $response = $this->postJson(
            route('brand.financial-insights.generate-campaign-report', $this->campaign),
            [
                'period_start' => now()->subDays(30)->format('Y-m-d'),
                'period_end' => now()->format('Y-m-d'),
                'run_ai_analysis' => false, // Skip AI for testing
            ]
        );

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $this->assertDatabaseHas('financial_reports', [
            'campaign_id' => $this->campaign->id,
            'brand_id' => $this->brand->id,
            'report_type' => 'campaign',
        ]);
    }

    /** @test */
    public function can_generate_brand_consolidated_report()
    {
        // Create multiple campaigns with financials
        $campaigns = Campaign::factory()->count(3)->create(['brand_id' => $this->brand->id]);
        
        foreach ($campaigns as $campaign) {
            CampaignFinancial::factory()->create([
                'campaign_id' => $campaign->id,
                'brand_id' => $this->brand->id,
            ]);
        }

        $this->actingAs($this->brand);

        $response = $this->postJson(
            route('brand.financial-insights.generate-brand-report'),
            [
                'period_start' => now()->subDays(30)->format('Y-m-d'),
                'period_end' => now()->format('Y-m-d'),
                'run_ai_analysis' => false,
            ]
        );

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $this->assertDatabaseHas('financial_reports', [
            'brand_id' => $this->brand->id,
            'report_type' => 'brand_consolidated',
        ]);
    }

    /** @test */
    public function brand_can_only_view_own_financial_reports()
    {
        $otherBrand = User::factory()->brand()->create();
        $report = FinancialReport::factory()->create([
            'brand_id' => $otherBrand->id,
        ]);

        $this->actingAs($this->brand);

        $response = $this->get(route('brand.financial-insights.show-report', $report));

        $response->assertStatus(403);
    }

    /** @test */
    public function can_generate_budget_scenarios()
    {
        $report = FinancialReport::factory()->create([
            'brand_id' => $this->brand->id,
            'campaign_id' => $this->campaign->id,
        ]);

        $this->actingAs($this->brand);

        $response = $this->postJson(
            route('brand.financial-insights.generate-scenarios', $report),
            ['total_budget' => 50000]
        );

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        // Should create 3 scenarios (conservative, balanced, aggressive)
        $this->assertEquals(3, BudgetScenario::where('brand_id', $this->brand->id)->count());
    }

    /** @test */
    public function can_apply_budget_scenario()
    {
        $scenario = BudgetScenario::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'draft',
            'is_applied' => false,
        ]);

        $this->actingAs($this->brand);

        $response = $this->postJson(
            route('brand.financial-insights.apply-scenario', $scenario)
        );

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $scenario->refresh();
        $this->assertTrue($scenario->is_applied);
        $this->assertEquals('active', $scenario->status);
    }

    /** @test */
    public function applying_scenario_deactivates_other_scenarios()
    {
        $scenario1 = BudgetScenario::factory()->create([
            'brand_id' => $this->brand->id,
            'is_applied' => true,
        ]);

        $scenario2 = BudgetScenario::factory()->create([
            'brand_id' => $this->brand->id,
            'is_applied' => false,
        ]);

        $this->actingAs($this->brand);

        $this->postJson(route('brand.financial-insights.apply-scenario', $scenario2));

        $scenario1->refresh();
        $scenario2->refresh();

        $this->assertFalse($scenario1->is_applied);
        $this->assertTrue($scenario2->is_applied);
    }

    /** @test */
    public function can_compare_two_financial_periods()
    {
        // Create financials for two different periods
        CampaignFinancial::factory()->create([
            'brand_id' => $this->brand->id,
            'period_start' => now()->subDays(60),
            'period_end' => now()->subDays(30),
        ]);

        CampaignFinancial::factory()->create([
            'brand_id' => $this->brand->id,
            'period_start' => now()->subDays(30),
            'period_end' => now(),
        ]);

        $this->actingAs($this->brand);

        $response = $this->postJson(
            route('brand.financial-insights.compare-periods'),
            [
                'period1_start' => now()->subDays(60)->format('Y-m-d'),
                'period1_end' => now()->subDays(30)->format('Y-m-d'),
                'period2_start' => now()->subDays(30)->format('Y-m-d'),
                'period2_end' => now()->format('Y-m-d'),
            ]
        );

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'success',
            'data' => [
                'period1',
                'period2',
                'comparison',
            ],
        ]);
    }

    /** @test */
    public function can_export_financial_report_as_pdf()
    {
        $report = FinancialReport::factory()->create([
            'brand_id' => $this->brand->id,
        ]);

        $this->actingAs($this->brand);

        $response = $this->get(route('brand.financial-insights.export-pdf', $report));

        $response->assertStatus(200);
        $response->assertHeader('content-type', 'application/pdf');
    }

    /** @test */
    public function export_increments_export_count()
    {
        $report = FinancialReport::factory()->create([
            'brand_id' => $this->brand->id,
            'export_count' => 0,
        ]);

        $this->actingAs($this->brand);

        $this->get(route('brand.financial-insights.export-pdf', $report));

        $report->refresh();
        $this->assertEquals(1, $report->export_count);
        $this->assertNotNull($report->last_exported_at);
    }

    /** @test */
    public function can_get_roi_trends_data()
    {
        CampaignFinancial::factory()->count(5)->create([
            'brand_id' => $this->brand->id,
        ]);

        $this->actingAs($this->brand);

        $response = $this->getJson(
            route('brand.financial-insights.roi-trends', [
                'period_start' => now()->subDays(30)->format('Y-m-d'),
                'period_end' => now()->format('Y-m-d'),
                'granularity' => 'weekly',
            ])
        );

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'success',
            'data' => [
                '*' => ['period', 'total_spend', 'total_revenue', 'roi'],
            ],
        ]);
    }

    /** @test */
    public function can_get_channel_performance_data()
    {
        $report = FinancialReport::factory()->create([
            'brand_id' => $this->brand->id,
            'report_type' => 'brand_consolidated',
            'spend_breakdown' => [
                'instagram' => 5000,
                'facebook' => 3000,
                'tiktok' => 2000,
            ],
            'revenue_breakdown' => [
                'instagram' => 8000,
                'facebook' => 4000,
                'tiktok' => 3000,
            ],
        ]);

        $this->actingAs($this->brand);

        $response = $this->getJson(
            route('brand.financial-insights.channel-performance', [
                'period_start' => $report->period_start->format('Y-m-d'),
                'period_end' => $report->period_end->format('Y-m-d'),
            ])
        );

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'success',
            'data' => [
                '*' => ['channel', 'spend', 'revenue', 'roi'],
            ],
        ]);
    }

    /** @test */
    public function financial_audit_log_is_created_on_data_update()
    {
        $this->actingAs($this->brand);

        $financial = CampaignFinancial::factory()->create([
            'campaign_id' => $this->campaign->id,
            'brand_id' => $this->brand->id,
        ]);

        $this->assertDatabaseHas('financial_audit_logs', [
            'user_id' => $this->brand->id,
            'auditable_type' => CampaignFinancial::class,
            'auditable_id' => $financial->id,
        ]);
    }

    /** @test */
    public function roi_status_is_calculated_correctly()
    {
        $excellentROI = CampaignFinancial::factory()->profitable()->create([
            'brand_id' => $this->brand->id,
        ]);

        $negativeROI = CampaignFinancial::factory()->unprofitable()->create([
            'brand_id' => $this->brand->id,
        ]);

        $this->assertEquals('positive', $excellentROI->getRoiStatus());
        $this->assertEquals('negative', $negativeROI->getRoiStatus());
    }

    /** @test */
    public function dashboard_data_includes_all_required_sections()
    {
        CampaignFinancial::factory()->count(3)->create([
            'brand_id' => $this->brand->id,
        ]);

        $dashboardData = $this->service->getDashboardData($this->brand);

        $this->assertArrayHasKey('overview', $dashboardData);
        $this->assertArrayHasKey('report', $dashboardData);
        $this->assertArrayHasKey('trends', $dashboardData);
        $this->assertArrayHasKey('top_campaigns', $dashboardData);
    }

    /** @test */
    public function insights_summary_returns_correct_structure()
    {
        $report = FinancialReport::factory()->create([
            'brand_id' => $this->brand->id,
            'ai_status' => 'completed',
            'actionable_insights' => [
                [
                    'type' => 'opportunity',
                    'title' => 'Test Insight',
                    'description' => 'Test description',
                ],
            ],
        ]);

        $insights = $this->service->getInsightsSummary($this->brand);

        $this->assertTrue($insights['has_insights']);
        $this->assertGreaterThan(0, $insights['insights_count']);
        $this->assertIsArray($insights['insights']);
    }
}