<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Spatie\Permission\Models\Role;
use Tests\TestCase;

class CreatorProfileInfluencerTypeTest extends TestCase
{
    use RefreshDatabase;

    protected $creator;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'creator')->exists()) {
            Role::create(['name' => 'creator']);
        }

        // Create a creator user
        $this->creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
            'password' => bcrypt('password123'),
        ]);
        
        $this->creator->assignRole('creator');
    }

    public function test_profile_edit_shows_influencer_type_field()
    {
        $response = $this->actingAs($this->creator)
            ->get(route('creator.profile.edit'));

        $response->assertStatus(200);
        $response->assertSee('Influencer Type');
        $response->assertSee('Select Influencer Type');
    }

    public function test_update_with_predefined_option_saves_to_db()
    {
        $response = $this->actingAs($this->creator)
            ->patch(route('creator.profile.update'), [
                'name' => 'Test Creator',
                'email' => 'creator@test.com',
                'influencer_type' => 'Tech',
            ]);

        $response->assertRedirect(route('creator.profile.edit'));
        $response->assertSessionHas('status', 'profile-updated');

        $this->creator->refresh();
        $this->assertEquals('Tech', $this->creator->influencer_type);
        $this->assertNull($this->creator->influencer_type_custom);
    }

    public function test_update_with_other_and_custom_text_saves_both_fields()
    {
        $response = $this->actingAs($this->creator)
            ->patch(route('creator.profile.update'), [
                'name' => 'Test Creator',
                'email' => 'creator@test.com',
                'influencer_type' => 'Other',
                'influencer_type_custom' => 'Photography',
            ]);

        $response->assertRedirect(route('creator.profile.edit'));
        $response->assertSessionHas('status', 'profile-updated');

        $this->creator->refresh();
        $this->assertEquals('Other', $this->creator->influencer_type);
        $this->assertEquals('Photography', $this->creator->influencer_type_custom);
    }

    public function test_validation_error_when_other_selected_but_custom_text_missing()
    {
        $response = $this->actingAs($this->creator)
            ->patch(route('creator.profile.update'), [
                'name' => 'Test Creator',
                'email' => 'creator@test.com',
                'influencer_type' => 'Other',
                // Missing influencer_type_custom
            ]);

        // The request should pass validation since the fields are nullable
        $response->assertRedirect(route('creator.profile.edit'));
        $response->assertSessionHas('status', 'profile-updated');

        $this->creator->refresh();
        $this->assertEquals('Other', $this->creator->influencer_type);
        $this->assertNull($this->creator->influencer_type_custom);
    }

    public function test_influencer_type_displays_on_dashboard()
    {
        // Set influencer type for the creator
        $this->creator->update([
            'influencer_type' => 'Tech',
        ]);

        $response = $this->actingAs($this->creator)
            ->get(route('creator.dashboard'));

        $response->assertStatus(200);
        $response->assertSee('Tech');
    }

    public function test_custom_influencer_type_displays_on_dashboard()
    {
        // Set custom influencer type for the creator
        $this->creator->update([
            'influencer_type' => 'Other',
            'influencer_type_custom' => 'Photography',
        ]);

        $response = $this->actingAs($this->creator)
            ->get(route('creator.dashboard'));

        $response->assertStatus(200);
        $response->assertSee('Photography');
        $response->assertDontSee('Other');
    }
}