<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandCreatorsAccessTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'superadmin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'superadmin', 'guard_name' => 'web']);
        }
    }

    public function test_brand_admin_can_access_creators_page()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.creators.index'));

        $response->assertStatus(200);
        $response->assertSee('Creators');
    }

    public function test_creator_cannot_access_brand_creators_page()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.creators.index'));

        // Should be redirected or get 403
        $response->assertStatus(403);
    }

    public function test_superadmin_can_access_creators_page()
    {
        $superadmin = User::factory()->create([
            'name' => 'Test SuperAdmin',
            'email' => 'superadmin@test.com',
        ]);
        $superadmin->assignRole('superadmin');

        $response = $this->actingAs($superadmin)
            ->get(route('brand.creators.index'));

        $response->assertStatus(200);
        $response->assertSee('Creators');
    }

    public function test_regular_user_cannot_access_brand_creators_page()
    {
        $user = User::factory()->create([
            'name' => 'Test User',
            'email' => 'user@test.com',
        ]);

        $response = $this->actingAs($user)
            ->get(route('brand.creators.index'));

        // Should be redirected or get 403
        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_brand_creators_page()
    {
        $response = $this->get(route('brand.creators.index'));

        // Should redirect to login
        $response->assertRedirect(route('login'));
    }
}