<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandAnalyticsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->exists()) {
            Role::create(['name' => 'brand_admin']);
        }
        
        if (!Role::where('name', 'creator')->exists()) {
            Role::create(['name' => 'creator']);
        }
    }

    public function test_brand_admin_can_access_analytics_dashboard()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::factory()->create();
        
        // Create a brand
        $brand = \App\Models\Brand::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'tenant_id' => $tenant->id,
            'brand_id' => $brand->id,
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.analytics.index'));

        $response->assertStatus(200);
        $response->assertSee('Analytics Dashboard');
    }

    public function test_brand_admin_can_get_analytics_data()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::factory()->create();
        
        // Create a brand
        $brand = \App\Models\Brand::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'tenant_id' => $tenant->id,
            'brand_id' => $brand->id,
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.analytics.data'));

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'labels',
            'data'
        ]);
    }

    public function test_creator_cannot_access_brand_analytics()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.analytics.index'));

        $response->assertStatus(403);

        $response = $this->actingAs($creator)
            ->get(route('brand.analytics.data'));

        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_analytics()
    {
        $response = $this->get(route('brand.analytics.index'));
        $response->assertRedirect(route('login'));

        $response = $this->get(route('brand.analytics.data'));
        $response->assertRedirect(route('login'));
    }
}