<?php

namespace Tests\Feature\Auth;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;
use Spatie\Permission\Models\Role;

class RoleRegistrationTest extends TestCase
{
    use RefreshDatabase;

    public function test_brand_registration(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test Brand',
            'email' => 'brand@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'brand',
            'company_name' => 'Test Brand Company',
            'brand_handle' => 'testbrand'
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'brand@example.com',
            'name' => 'Test Brand'
        ]);

        $user = User::where('email', 'brand@example.com')->first();
        $this->assertTrue($user->hasRole('brand'));
        $this->assertAuthenticatedAs($user);
        $response->assertRedirect(route('brand.dashboard'));
    }

    public function test_agency_registration(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test Agency',
            'email' => 'agency@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'agency',
            'company_name' => 'Test Agency Company'
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'agency@example.com',
            'name' => 'Test Agency'
        ]);

        $user = User::where('email', 'agency@example.com')->first();
        $this->assertTrue($user->hasRole('agency'));
        $this->assertAuthenticatedAs($user);
        $response->assertRedirect(route('agency.dashboard'));
    }

    public function test_creator_registration(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test Creator',
            'email' => 'creator@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'creator'
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'creator@example.com',
            'name' => 'Test Creator'
        ]);

        $user = User::where('email', 'creator@example.com')->first();
        $this->assertTrue($user->hasRole('creator'));
        $this->assertAuthenticatedAs($user);
        $response->assertRedirect(route('creator.dashboard'));
    }

    public function test_user_registration(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test User',
            'email' => 'user@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'user'
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'user@example.com',
            'name' => 'Test User'
        ]);

        $user = User::where('email', 'user@example.com')->first();
        $this->assertTrue($user->hasRole('user'));
        $this->assertAuthenticatedAs($user);
        $response->assertRedirect(route('dashboard', absolute: false));
    }

    public function test_registration_fails_with_invalid_role(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test Invalid',
            'email' => 'invalid@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'invalid_role'
        ]);

        $response->assertSessionHasErrors(['role']);
        $this->assertGuest();
    }

    public function test_registration_fails_with_duplicate_email(): void
    {
        // Create a user first
        User::factory()->create([
            'email' => 'duplicate@example.com'
        ]);

        $response = $this->post('/register', [
            'name' => 'Test Duplicate',
            'email' => 'duplicate@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'user'
        ]);

        $response->assertSessionHasErrors(['email']);
        $this->assertGuest();
    }

    public function test_registration_logs_audit_event(): void
    {
        $this->post('/register', [
            'name' => 'Audit Test User',
            'email' => 'audit@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
            'role' => 'user'
        ]);

        $this->assertDatabaseHas('audit_logs', [
            'action' => 'user_registered_public',
            'meta->role' => 'user'
        ]);
    }
}