@extends('adminlte::page')

@section('title', 'Campaign Planner')

@section('content')
<div class="content-header">
    <div class="container-fluid">
        <div class="row mb-2">
            <div class="col-sm-6">
                <h1 class="m-0">Campaign Planner: {{ $campaign->name }}</h1>
            </div>
            <div class="col-sm-6">
                <ol class="breadcrumb float-sm-right">
                    <li class="breadcrumb-item"><a href="{{ route('creator.dashboard') }}">Home</a></li>
                    <li class="breadcrumb-item"><a href="{{ route('creator.campaigns.index') }}">Campaigns</a></li>
                    <li class="breadcrumb-item"><a href="{{ route('creator.campaigns.show', $campaign) }}">Details</a></li>
                    <li class="breadcrumb-item active">Planner</li>
                </ol>
            </div>
        </div>
    </div>
</div>

<div class="content">
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Campaign Overview</h3>
                        <div class="card-tools">
                            <button type="button" class="btn btn-tool" data-card-widget="collapse">
                                <i class="fas fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <th>Name:</th>
                                        <td>{{ $campaign->name }}</td>
                                    </tr>
                                    <tr>
                                        <th>Status:</th>
                                        <td>
                                            @switch($campaign->status)
                                                @case('draft')
                                                    <span class="badge badge-secondary">Draft</span>
                                                    @break
                                                @case('active')
                                                    <span class="badge badge-success">Active</span>
                                                    @break
                                                @case('paused')
                                                    <span class="badge badge-warning">Paused</span>
                                                    @break
                                                @case('completed')
                                                    <span class="badge badge-info">Completed</span>
                                                    @break
                                                @case('cancelled')
                                                    <span class="badge badge-danger">Cancelled</span>
                                                    @break
                                                @default
                                                    <span class="badge badge-secondary">{{ ucfirst($campaign->status) }}</span>
                                            @endswitch
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>Budget:</th>
                                        <td>
                                            {{ $campaign->budget ? '$' . number_format($campaign->budget, 2) : 'N/A' }}
                                            @if($campaign->encrypted_budget)
                                                <span class="badge badge-success">Encrypted</span>
                                            @endif
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>Currency:</th>
                                        <td>{{ $campaign->currency }}</td>
                                    </tr>
                                    <tr>
                                        <th>Dates:</th>
                                        <td>
                                            @if($campaign->start_date && $campaign->end_date)
                                                {{ $campaign->start_date->format('M j, Y') }} - {{ $campaign->end_date->format('M j, Y') }}
                                            @else
                                                N/A
                                            @endif
                                        </td>
                                    </tr>
                                </table>
                            </div>
                            
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <th>Objective:</th>
                                        <td>{{ $campaign->objective ?? 'N/A' }}</td>
                                    </tr>
                                    <tr>
                                        <th>Deliverables:</th>
                                        <td>{{ $campaign->deliverables ?? 'N/A' }}</td>
                                    </tr>
                                    <tr>
                                        <th>Audience:</th>
                                        <td>{{ $campaign->audience ?? 'N/A' }}</td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <div class="row mt-3">
                            <div class="col-12">
                                <button type="button" class="btn btn-primary" id="requestAiPlanning">
                                    <i class="fas fa-robot"></i> Request AI Planning Suggestions
                                </button>
                                <button type="button" class="btn btn-success" id="exportCampaign">
                                    <i class="fas fa-file-export"></i> Export Campaign Summary
                                </button>
                                <a href="{{ route('creator.campaigns.show', $campaign) }}" class="btn btn-secondary">
                                    <i class="fas fa-arrow-left"></i> Back to Campaign Details
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Milestones -->
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">My Milestones</h3>
                        <div class="card-tools">
                            <button type="button" class="btn btn-tool" data-card-widget="collapse">
                                <i class="fas fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped" id="milestonesTable">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Due Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($campaign->milestones as $milestone)
                                        @if($milestone->assigned_creator_id == auth()->id())
                                            <tr data-id="{{ $milestone->id }}">
                                                <td>{{ $milestone->name }}</td>
                                                <td>{{ $milestone->description ?? 'N/A' }}</td>
                                                <td>{{ $milestone->due_date ? $milestone->due_date->format('M j, Y') : 'N/A' }}</td>
                                                <td>
                                                    @switch($milestone->status)
                                                        @case('pending')
                                                            <span class="badge badge-secondary">Pending</span>
                                                            @break
                                                        @case('in_progress')
                                                            <span class="badge badge-warning">In Progress</span>
                                                            @break
                                                        @case('completed')
                                                            <span class="badge badge-success">Completed</span>
                                                            @break
                                                        @default
                                                            <span class="badge badge-secondary">{{ ucfirst($milestone->status) }}</span>
                                                    @endswitch
                                                </td>
                                                <td>
                                                    <button type="button" class="btn btn-info btn-sm edit-milestone" data-id="{{ $milestone->id }}">
                                                        <i class="fas fa-edit"></i> Edit
                                                    </button>
                                                    <button type="button" class="btn btn-danger btn-sm delete-milestone" data-id="{{ $milestone->id }}">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>
                                                </td>
                                            </tr>
                                        @endif
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <button type="button" class="btn btn-primary" id="addMilestone">
                                <i class="fas fa-plus"></i> Add Milestone
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- AI Planning Suggestions -->
                <div class="card" id="aiSuggestionsCard" style="display: none;">
                    <div class="card-header">
                        <h3 class="card-title">AI Planning Suggestions</h3>
                        <div class="card-tools">
                            <button type="button" class="btn btn-tool" data-card-widget="collapse">
                                <i class="fas fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <div id="aiSuggestionsContent"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Milestone Modal -->
<div class="modal fade" id="milestoneModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="milestoneModalLabel">Milestone</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="milestoneForm">
                    <input type="hidden" id="milestoneId" name="id">
                    <div class="form-group">
                        <label for="milestoneName">Name</label>
                        <input type="text" class="form-control" id="milestoneName" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="milestoneDescription">Description</label>
                        <textarea class="form-control" id="milestoneDescription" name="description" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <label for="dueDate">Due Date</label>
                        <input type="date" class="form-control" id="dueDate" name="due_date">
                    </div>
                    <div class="form-group">
                        <label for="milestoneStatus">Status</label>
                        <select class="form-control" id="milestoneStatus" name="status" required>
                            <option value="pending">Pending</option>
                            <option value="in_progress">In Progress</option>
                            <option value="completed">Completed</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveMilestone">Save</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Request AI Planning
    document.getElementById('requestAiPlanning').addEventListener('click', function() {
        if (confirm('Are you sure you want to request AI planning suggestions for this campaign?')) {
            fetch('{{ route("creator.campaigns.ai-planning.request", $campaign) }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('AI planning request submitted successfully! Check back later for suggestions.');
                    // Reload page to show updated data
                    location.reload();
                } else {
                    alert('Error submitting AI planning request.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error submitting AI planning request.');
            });
        }
    });
    
    // Export Campaign
    document.getElementById('exportCampaign').addEventListener('click', function() {
        if (confirm('Are you sure you want to export this campaign as a summary?')) {
            fetch('{{ route("creator.campaigns.export", $campaign) }}', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Campaign exported successfully!');
                } else {
                    alert('Error exporting campaign.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error exporting campaign.');
            });
        }
    });
    
    // Add Milestone
    document.getElementById('addMilestone').addEventListener('click', function() {
        // Reset form
        document.getElementById('milestoneId').value = '';
        document.getElementById('milestoneName').value = '';
        document.getElementById('milestoneDescription').value = '';
        document.getElementById('dueDate').value = '';
        document.getElementById('milestoneStatus').value = 'pending';
        
        // Show modal
        $('#milestoneModal').modal('show');
    });
    
    // Edit Milestone
    document.querySelectorAll('.edit-milestone').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const row = document.querySelector(`tr[data-id="${id}"]`);
            const cells = row.querySelectorAll('td');
            
            // Fill form with existing data
            document.getElementById('milestoneId').value = id;
            document.getElementById('milestoneName').value = cells[0].textContent;
            document.getElementById('milestoneDescription').value = cells[1].textContent === 'N/A' ? '' : cells[1].textContent;
            document.getElementById('dueDate').value = cells[2].textContent === 'N/A' ? '' : new Date(cells[2].textContent).toISOString().split('T')[0];
            
            // Set status
            const statusText = cells[3].textContent.trim();
            let statusValue = 'pending';
            if (statusText === 'Pending') statusValue = 'pending';
            else if (statusText === 'In Progress') statusValue = 'in_progress';
            else if (statusText === 'Completed') statusValue = 'completed';
            document.getElementById('milestoneStatus').value = statusValue;
            
            // Show modal
            $('#milestoneModal').modal('show');
        });
    });
    
    // Delete Milestone
    document.querySelectorAll('.delete-milestone').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            
            if (confirm('Are you sure you want to delete this milestone?')) {
                fetch(`/creator/milestones/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Remove row from table
                        document.querySelector(`tr[data-id="${id}"]`).remove();
                        alert('Milestone deleted successfully!');
                    } else {
                        alert('Error deleting milestone.');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error deleting milestone.');
                });
            }
        });
    });
    
    // Save Milestone
    document.getElementById('saveMilestone').addEventListener('click', function() {
        const id = document.getElementById('milestoneId').value;
        const url = id ? `/creator/milestones/${id}` : `/creator/campaigns/{{ $campaign->id }}/milestones`;
        const method = id ? 'PUT' : 'POST';
        
        const formData = new FormData(document.getElementById('milestoneForm'));
        
        fetch(url, {
            method: method,
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Close modal
                $('#milestoneModal').modal('hide');
                
                // Reload page to show updated data
                location.reload();
            } else {
                alert('Error saving milestone.');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error saving milestone.');
        });
    });
    
    // Check for AI suggestions
    checkForAiSuggestions();
});

function checkForAiSuggestions() {
    // Look for completed AI tasks for this campaign
    fetch('/api/ai-tasks?campaign_id={{ $campaign->id }}&type=creator_campaign_planning&status=completed')
        .then(response => response.json())
        .then(data => {
            if (data.length > 0) {
                const aiTask = data[0];
                displayAiSuggestions(aiTask.result_meta);
            }
        })
        .catch(error => {
            console.error('Error checking for AI suggestions:', error);
        });
}

function displayAiSuggestions(suggestions) {
    const contentDiv = document.getElementById('aiSuggestionsContent');
    const card = document.getElementById('aiSuggestionsCard');
    
    // Create HTML for suggestions
    let html = '<h4>Content Creation Suggestions</h4>';
    
    // Milestones suggestions
    if (suggestions.milestones && suggestions.milestones.length > 0) {
        html += '<h5>Recommended Milestones</h5>';
        html += '<ul>';
        suggestions.milestones.forEach(milestone => {
            html += `<li><strong>${milestone.name}</strong>: ${milestone.description} (Due: ${milestone.due_date})</li>`;
        });
        html += '</ul>';
    }
    
    // Posting schedule suggestions
    if (suggestions.posting_schedule && suggestions.posting_schedule.length > 0) {
        html += '<h5>Recommended Posting Schedule</h5>';
        html += '<div class="table-responsive">';
        html += '<table class="table table-striped">';
        html += '<thead><tr><th>Date</th><th>Platform</th><th>Content Type</th><th>Theme</th></tr></thead>';
        html += '<tbody>';
        suggestions.posting_schedule.forEach(item => {
            html += `<tr><td>${item.date}</td><td>${item.platform}</td><td>${item.content_type}</td><td>${item.theme}</td></tr>`;
        });
        html += '</tbody></table></div>';
    }
    
    contentDiv.innerHTML = html;
    card.style.display = 'block';
}
</script>
@endsection