// Function to render the views chart
function renderViewsChart(canvasId, labels, data) {
    const ctx = document.getElementById(canvasId).getContext('2d');
    
    // Destroy existing chart if it exists
    if (window.viewsChart) {
        window.viewsChart.destroy();
    }
    
    // Create new chart
    window.viewsChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Views',
                data: data,
                borderColor: 'rgba(60, 141, 188, 1)',
                backgroundColor: 'rgba(60, 141, 188, 0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            if (value >= 1000) {
                                return (value / 1000).toFixed(1) + 'k';
                            }
                            return value;
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                    callbacks: {
                        label: function(context) {
                            return context.dataset.label + ': ' + context.parsed.y.toLocaleString();
                        }
                    }
                }
            }
        }
    });
}

// Function to handle sync button click
function handleSyncButtonClick() {
    const syncForm = document.getElementById('syncForm');
    const syncButton = document.getElementById('syncButton');
    
    if (!syncForm || !syncButton) {
        return;
    }
    
    syncForm.addEventListener('submit', function(e) {
        const originalText = syncButton.innerHTML;
        
        // Show loading state
        syncButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Syncing...';
        syncButton.disabled = true;
        
        // Show toast notification
        showSyncToast('Sync started! This may take a few moments.', 'info');
        
        // Revert button after 3 seconds
        setTimeout(function() {
            syncButton.innerHTML = originalText;
            syncButton.disabled = false;
            
            // Show success toast
            showSyncToast('Sync completed successfully!', 'success');
        }, 3000);
    });
}

// Function to show toast notifications
function showSyncToast(message, type = 'info') {
    // Create toast container if it doesn't exist
    let toastContainer = document.getElementById('syncToastContainer');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'syncToastContainer';
        toastContainer.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
        `;
        document.body.appendChild(toastContainer);
    }
    
    // Create toast element
    const toast = document.createElement('div');
    toast.className = `alert alert-${type} alert-dismissible fade show`;
    toast.style.cssText = `
        min-width: 300px;
        margin-bottom: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    `;
    toast.innerHTML = `
        ${message}
        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
            <span aria-hidden="true">&times;</span>
        </button>
    `;
    
    // Add to container
    toastContainer.appendChild(toast);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (toast.parentNode) {
            toast.parentNode.removeChild(toast);
        }
    }, 5000);
}

// Export the function for use in other modules
export { renderViewsChart, handleSyncButtonClick, showSyncToast };