<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class AgencyDemoSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get the demo tenant
        $tenant = \App\Models\Tenant::where('domain', 'demo.local')->first();
        
        if (!$tenant) {
            throw new \Exception('Demo tenant must be created first. Please run TenantsSeeder first.');
        }
        
        // Get brand for the demo tenant
        $brand = \App\Models\Brand::where('tenant_id', $tenant->id)->first();
        
        if (!$brand) {
            throw new \Exception('Brand for demo tenant must be created first. Please run BrandDemoSeeder first.');
        }
        
        // Create Agency 1
        $agency1 = \App\Models\User::firstOrCreate([
            'email' => 'agency1@demo.local'
        ], [
            'tenant_id' => $tenant->id,
            'name' => 'Agency One',
            'password' => bcrypt('password123'),
        ]);
        
        // Assign agency_admin role if not already assigned
        if (!$agency1->hasRole('agency_admin')) {
            $agency1->assignRole('agency_admin');
        }
        
        // Create Agency 2
        $agency2 = \App\Models\User::firstOrCreate([
            'email' => 'agency2@demo.local'
        ], [
            'tenant_id' => $tenant->id,
            'name' => 'Agency Two',
            'password' => bcrypt('password123'),
        ]);
        
        // Assign agency_admin role if not already assigned
        if (!$agency2->hasRole('agency_admin')) {
            $agency2->assignRole('agency_admin');
        }
        
        // Link agencies to brand using the agency_brand pivot table
        // Agency 1 linked to brand
        $agencyBrand1 = DB::table('agency_brand')->where([
            'agency_id' => $agency1->id,
            'brand_id' => $brand->id,
        ])->first();
        
        if (!$agencyBrand1) {
            DB::table('agency_brand')->insert([
                'agency_id' => $agency1->id,
                'brand_id' => $brand->id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
        
        // Agency 2 linked to brand
        $agencyBrand2 = DB::table('agency_brand')->where([
            'agency_id' => $agency2->id,
            'brand_id' => $brand->id,
        ])->first();
        
        if (!$agencyBrand2) {
            DB::table('agency_brand')->insert([
                'agency_id' => $agency2->id,
                'brand_id' => $brand->id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
        
        // Create 2 campaigns for Agency 1
        \App\Models\Campaign::firstOrCreate([
            'agency_id' => $agency1->id,
            'brand_id' => $brand->id,
            'name' => 'Summer Promotion Campaign',
        ], [
            'start_date' => now()->subDays(30),
           'end_date' => now()->addDays(30),
           'budget' => 5000.00,
           'status' => 'active',
           'created_by' => $agency1->id,
       ]);
        
        \App\Models\Campaign::firstOrCreate([
            'agency_id' => $agency1->id,
            'brand_id' => $brand->id,
            'name' => 'Holiday Marketing Drive',
        ], [
            'start_date' => now()->addDays(10),
           'end_date' => now()->addDays(70),
           'budget' => 10000.00,
           'status' => 'planned',
           'created_by' => $agency1->id,
       ]);
        
        // Create 2 campaigns for Agency 2
        \App\Models\Campaign::firstOrCreate([
            'agency_id' => $agency2->id,
            'brand_id' => $brand->id,
            'name' => 'Product Launch Campaign',
        ], [
            'start_date' => now()->subDays(15),
           'end_date' => now()->addDays(45),
           'budget' => 7500.00,
           'status' => 'active',
           'created_by' => $agency2->id,
       ]);
        
        \App\Models\Campaign::firstOrCreate([
            'agency_id' => $agency2->id,
            'brand_id' => $brand->id,
            'name' => 'Brand Awareness Initiative',
        ], [
            'start_date' => now()->subDays(60),
           'end_date' => now()->subDays(30),
           'budget' => 3000.00,
           'status' => 'completed',
           'created_by' => $agency2->id,
       ]);
    }
}