<?php

namespace Database\Factories;

use App\Models\SocialAccount;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Facades\Crypt;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\SocialAccount>
 */
class SocialAccountFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'user_id' => null,
            'brand_id' => null,
            'tenant_id' => null,
            'provider' => $this->faker->randomElement(['youtube', 'instagram', 'twitter', 'linkedin', 'tiktok']),
            'provider_user_id' => $this->faker->unique()->uuid,
            'access_token_encrypted' => Crypt::encryptString($this->faker->sha256),
            'refresh_token_encrypted' => Crypt::encryptString($this->faker->sha256),
            'token_expires_at' => $this->faker->dateTimeBetween('+1 month', '+1 year'),
            'meta' => [
                'username' => $this->faker->userName,
                'name' => $this->faker->name,
            ],
            'connected_at' => $this->faker->dateTimeBetween('-1 year', 'now'),
            'last_synced_at' => $this->faker->dateTimeBetween('-1 month', 'now'),
            'is_demo' => false,
        ];
    }

    /**
     * Indicate that the account is a demo account.
     *
     * @return static
     */
    public function demo()
    {
        return $this->state(function (array $attributes) {
            return [
                'access_token_encrypted' => null,
                'refresh_token_encrypted' => null,
                'token_expires_at' => null,
                'is_demo' => true,
            ];
        });
    }

    /**
     * Indicate that the token is expired.
     *
     * @return static
     */
    public function expired()
    {
        return $this->state(function (array $attributes) {
            return [
                'token_expires_at' => $this->faker->dateTimeBetween('-1 year', '-1 day'),
            ];
        });
    }
    
    /**
     * Indicate that the account belongs to a brand.
     *
     * @param int $brandId
     * @param int $tenantId
     * @return static
     */
    public function forBrand($brandId, $tenantId)
    {
        return $this->state(function (array $attributes) use ($brandId, $tenantId) {
            return [
                'user_id' => null,
                'brand_id' => $brandId,
                'tenant_id' => $tenantId,
            ];
        });
    }
}
