<?php

namespace Database\Factories;

use App\Models\LocalizedContent;
use App\Models\Brand;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\LocalizedContent>
 */
class LocalizedContentFactory extends Factory
{
    protected $model = LocalizedContent::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $locales = ['es-ES', 'fr-FR', 'de-DE', 'it-IT', 'pt-BR', 'ja-JP', 'zh-CN'];
        $tones = ['formal', 'friendly', 'neutral'];
        $statuses = ['pending', 'generated', 'approved', 'rejected', 'edited'];
        $contentTypes = ['ad', 'post', 'proposal', 'campaign'];

        return [
            'brand_id' => Brand::factory(),
            'content_type' => $this->faker->randomElement($contentTypes),
            'content_id' => $this->faker->numberBetween(1, 1000),
            'locale_code' => $this->faker->randomElement($locales),
            'source_text' => $this->faker->paragraph(),
            'localized_text' => $this->faker->paragraph(),
            'cultural_notes' => $this->faker->sentence(),
            'tone' => $this->faker->randomElement($tones),
            'ai_job_id' => 'ai_trans_' . uniqid() . '_' . time(),
            'status' => $this->faker->randomElement($statuses),
            'created_by' => User::factory(),
            'approved_by' => null,
            'approved_at' => null,
        ];
    }

    /**
     * Indicate that the localization is approved.
     */
    public function approved(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'approved',
            'approved_by' => User::factory(),
            'approved_at' => now(),
        ]);
    }

    /**
     * Indicate that the localization is pending.
     */
    public function pending(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'pending',
            'approved_by' => null,
            'approved_at' => null,
        ]);
    }

    /**
     * Indicate that the localization is generated.
     */
    public function generated(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'generated',
            'approved_by' => null,
            'approved_at' => null,
        ]);
    }

    /**
     * Indicate that the localization is rejected.
     */
    public function rejected(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'rejected',
            'approved_by' => null,
            'approved_at' => null,
        ]);
    }

    /**
     * Indicate that the localization is edited.
     */
    public function edited(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'edited',
        ]);
    }

    /**
     * Set specific locale.
     */
    public function locale(string $locale): static
    {
        return $this->state(fn (array $attributes) => [
            'locale_code' => $locale,
        ]);
    }

    /**
     * Set specific tone.
     */
    public function tone(string $tone): static
    {
        return $this->state(fn (array $attributes) => [
            'tone' => $tone,
        ]);
    }

    /**
     * Set specific content type.
     */
    public function contentType(string $type): static
    {
        return $this->state(fn (array $attributes) => [
            'content_type' => $type,
        ]);
    }
}