<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;

class PIIRedactionService
{
    /**
     * PII patterns for detection and redaction.
     */
    protected array $piiPatterns = [
        'email' => [
            'pattern' => '/\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b/',
            'replacement' => '[EMAIL_REDACTED]',
        ],
        'phone' => [
            'pattern' => '/\b(?:\+?1[-.]?)?\(?([0-9]{3})\)?[-.]?([0-9]{3})[-.]?([0-9]{4})\b/',
            'replacement' => '[PHONE_REDACTED]',
        ],
        'ssn' => [
            'pattern' => '/\b\d{3}-\d{2}-\d{4}\b/',
            'replacement' => '[SSN_REDACTED]',
        ],
        'credit_card' => [
            'pattern' => '/\b(?:\d{4}[-\s]?){3}\d{4}\b/',
            'replacement' => '[CARD_REDACTED]',
        ],
        'ip_address' => [
            'pattern' => '/\b(?:\d{1,3}\.){3}\d{1,3}\b/',
            'replacement' => '[IP_REDACTED]',
        ],
        'address' => [
            'pattern' => '/\b\d+\s+[\w\s]+(?:street|st|avenue|ave|road|rd|highway|hwy|square|sq|trail|trl|drive|dr|court|ct|parkway|pkwy|circle|cir|boulevard|blvd)\b/i',
            'replacement' => '[ADDRESS_REDACTED]',
        ],
    ];

    /**
     * Redact PII from text.
     *
     * @param string $text The text to redact
     * @param array|null $customPatterns Additional custom patterns
     * @return array ['redacted_text' => string, 'detected_pii' => array]
     */
    public function redact(string $text, ?array $customPatterns = null): array
    {
        $redactedText = $text;
        $detectedPII = [];

        // Merge custom patterns if provided
        $patterns = $this->piiPatterns;
        if ($customPatterns) {
            $patterns = array_merge($patterns, $customPatterns);
        }

        // Apply each pattern
        foreach ($patterns as $type => $config) {
            $matches = [];
            if (preg_match_all($config['pattern'], $redactedText, $matches)) {
                $detectedPII[$type] = count($matches[0]);
                $redactedText = preg_replace(
                    $config['pattern'],
                    $config['replacement'],
                    $redactedText
                );
            }
        }

        Log::info('PII Redaction performed', [
            'original_length' => strlen($text),
            'redacted_length' => strlen($redactedText),
            'detected_pii' => $detectedPII,
        ]);

        return [
            'redacted_text' => $redactedText,
            'detected_pii' => $detectedPII,
        ];
    }

    /**
     * Check if text contains PII.
     *
     * @param string $text The text to check
     * @return bool
     */
    public function containsPII(string $text): bool
    {
        foreach ($this->piiPatterns as $config) {
            if (preg_match($config['pattern'], $text)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Detect PII types in text without redacting.
     *
     * @param string $text The text to analyze
     * @return array Array of detected PII types
     */
    public function detectPIITypes(string $text): array
    {
        $detected = [];

        foreach ($this->piiPatterns as $type => $config) {
            if (preg_match($config['pattern'], $text)) {
                $detected[] = $type;
            }
        }

        return $detected;
    }

    /**
     * Get all available PII patterns.
     *
     * @return array
     */
    public function getPatterns(): array
    {
        return $this->piiPatterns;
    }

    /**
     * Add custom PII pattern.
     *
     * @param string $name Pattern name
     * @param string $pattern Regex pattern
     * @param string $replacement Replacement text
     */
    public function addPattern(string $name, string $pattern, string $replacement): void
    {
        $this->piiPatterns[$name] = [
            'pattern' => $pattern,
            'replacement' => $replacement,
        ];
    }
}