<?php

namespace App\Services;

class AnalyticsService
{
    /**
     * Track a custom event
     */
    public function trackEvent(string $event, array $properties = []): void
    {
        // Add user context
        $properties['user_id'] = auth()->id();
        $properties['timestamp'] = now()->toIso8601String();
        $properties['url'] = request()->fullUrl();
        $properties['user_agent'] = request()->userAgent();
        $properties['ip_address'] = request()->ip();
        
        // Track in Google Analytics 4 (client-side via gtag.js)
        // Track in Facebook Pixel (client-side via fbq())
        
        // Store in database for custom analytics (optional)
        $this->storeCustomEvent($event, $properties);
    }

    /**
     * Track conversion event
     */
    public function trackConversion(string $type, float $value, string $currency = 'INR'): void
    {
        $this->trackEvent('conversion', [
            'conversion_type' => $type,
            'value' => $value,
            'currency' => $currency
        ]);
    }

    /**
     * Track pricing page view
     */
    public function trackPricingView(string $role = null, string $interval = 'monthly'): void
    {
        $this->trackEvent('pricing_view', [
            'role' => $role,
            'interval' => $interval
        ]);
    }

    /**
     * Track plan CTA click
     */
    public function trackPlanClick(string $planSlug, string $action = 'subscribe'): void
    {
        $this->trackEvent('plan_cta_click', [
            'plan' => $planSlug,
            'action' => $action
        ]);
    }

    /**
     * Track checkout initiation
     */
    public function trackCheckoutInitiated(int $planId, float $amount): void
    {
        $this->trackEvent('checkout_initiated', [
            'plan_id' => $planId,
            'amount' => $amount
        ]);
    }

    /**
     * Track payment success
     */
    public function trackPaymentSuccess(int $subscriptionId, float $amount): void
    {
        $this->trackConversion('subscription', $amount);
        
        $this->trackEvent('payment_success', [
            'subscription_id' => $subscriptionId,
            'amount' => $amount
        ]);
    }

    /**
     * Track payment failure
     */
    public function trackPaymentFailed(string $reason): void
    {
        $this->trackEvent('payment_failed', [
            'reason' => $reason
        ]);
    }

    /**
     * Track trial started
     */
    public function trackTrialStarted(int $planId): void
    {
        $this->trackEvent('trial_started', [
            'plan_id' => $planId
        ]);
    }

    /**
     * Track feature exploration
     */
    public function trackFeatureExplored(string $featureName): void
    {
        $this->trackEvent('feature_explored', [
            'feature' => $featureName
        ]);
    }

    /**
     * Track FAQ interaction
     */
    public function trackFaqOpened(string $question): void
    {
        $this->trackEvent('faq_opened', [
            'question' => $question
        ]);
    }

    /**
     * Store custom event in database (optional)
     */
    protected function storeCustomEvent(string $event, array $properties): void
    {
        // Optional: Store events in database for custom analytics
        // Uncomment if you want to store events
        /*
        \DB::table('analytics_events')->insert([
            'event' => $event,
            'properties' => json_encode($properties),
            'user_id' => auth()->id(),
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'created_at' => now(),
        ]);
        */
    }

    /**
     * Get conversion funnel data
     */
    public function getConversionFunnel(string $startDate, string $endDate): array
    {
        // This would query your analytics database
        // For now, return sample data structure
        return [
            'landing_views' => 0,
            'pricing_views' => 0,
            'checkout_initiated' => 0,
            'payment_success' => 0,
            'trial_started' => 0,
        ];
    }
}