<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Brand;
use App\Models\ReputationScore;
use App\Models\ReputationDriver;
use App\Models\ReputationRecommendation;

class ReputationScorePolicy
{
    /**
     * Determine if the user can view reputation scores.
     */
    public function view(User $user, ReputationScore|Brand $resource): bool
    {
        // Brand admins can view their own brand's reputation
        if ($user->role === 'brand' && $user->brand_id) {
            if ($resource instanceof ReputationScore) {
                return $user->brand_id === $resource->brand_id;
            }
            if ($resource instanceof Brand) {
                return $user->brand_id === $resource->id;
            }
        }

        // Platform admins can view all
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can calculate reputation scores.
     */
    public function calculate(User $user, Brand $brand): bool
    {
        // Brand admins can calculate for their brand
        if ($user->role === 'brand' && $user->brand_id === $brand->id) {
            return true;
        }

        // Platform admins can calculate for any brand
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can update reputation scores.
     */
    public function update(User $user, ReputationScore $reputationScore): bool
    {
        // Brand admins can update their own scores (e.g., dismiss alerts)
        if ($user->role === 'brand' && $user->brand_id === $reputationScore->brand_id) {
            return true;
        }

        // Platform admins can update any score
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can resolve drivers.
     */
    public function resolve(User $user, ReputationDriver $driver): bool
    {
        // Brand admins can resolve their own drivers
        if ($user->role === 'brand' && $user->brand_id === $driver->brand_id) {
            return true;
        }

        // Platform admins can resolve any driver
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can view drivers.
     */
    public function viewDriver(User $user, ReputationDriver $driver): bool
    {
        // Brand admins can view their own drivers
        if ($user->role === 'brand' && $user->brand_id === $driver->brand_id) {
            return true;
        }

        // Platform admins can view any driver
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can update recommendations.
     */
    public function updateRecommendation(User $user, ReputationRecommendation $recommendation): bool
    {
        // Brand admins can update their own recommendations
        if ($user->role === 'brand' && $user->brand_id === $recommendation->brand_id) {
            return true;
        }

        // Platform admins can update any recommendation
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can delete reputation scores.
     */
    public function delete(User $user, ReputationScore $reputationScore): bool
    {
        // Only platform admins can delete scores
        return $user->role === 'admin';
    }

    /**
     * Determine if the user can export reputation data.
     */
    public function export(User $user, ReputationScore $reputationScore): bool
    {
        // Brand admins can export their own data
        if ($user->role === 'brand' && $user->brand_id === $reputationScore->brand_id) {
            return true;
        }

        // Platform admins can export any data
        return $user->role === 'admin';
    }
}