<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SupportAnalytics extends Model
{
    use HasFactory;

    protected $fillable = [
        'brand_id',
        'date',
        'total_conversations',
        'resolved_by_bot',
        'escalated_to_human',
        'total_messages',
        'avg_confidence_score',
        'avg_resolution_time_minutes',
        'avg_satisfaction_rating',
        'positive_feedback_count',
        'negative_feedback_count',
        'top_queries',
        'low_confidence_topics',
    ];

    protected $casts = [
        'date' => 'date',
        'avg_confidence_score' => 'decimal:2',
        'avg_resolution_time_minutes' => 'decimal:2',
        'avg_satisfaction_rating' => 'decimal:2',
        'top_queries' => 'array',
        'low_confidence_topics' => 'array',
    ];

    /**
     * Get the brand that owns the analytics.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Calculate bot resolution rate.
     */
    public function getBotResolutionRateAttribute(): float
    {
        if ($this->total_conversations === 0) {
            return 0;
        }

        return round(($this->resolved_by_bot / $this->total_conversations) * 100, 2);
    }

    /**
     * Calculate escalation rate.
     */
    public function getEscalationRateAttribute(): float
    {
        if ($this->total_conversations === 0) {
            return 0;
        }

        return round(($this->escalated_to_human / $this->total_conversations) * 100, 2);
    }

    /**
     * Calculate positive feedback rate.
     */
    public function getPositiveFeedbackRateAttribute(): float
    {
        $totalFeedback = $this->positive_feedback_count + $this->negative_feedback_count;
        
        if ($totalFeedback === 0) {
            return 0;
        }

        return round(($this->positive_feedback_count / $totalFeedback) * 100, 2);
    }

    /**
     * Get average messages per conversation.
     */
    public function getAvgMessagesPerConversationAttribute(): float
    {
        if ($this->total_conversations === 0) {
            return 0;
        }

        return round($this->total_messages / $this->total_conversations, 2);
    }

    /**
     * Scope to get analytics for date range.
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    /**
     * Scope to get analytics for specific brand.
     */
    public function scopeForBrand($query, int $brandId)
    {
        return $query->where('brand_id', $brandId);
    }
}