<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class SocialSyncError extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'social_account_id',
        'error_text',
        'response_code',
        'meta',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'meta' => 'array',
        'created_at' => 'datetime',
    ];

    /**
     * Get the social account associated with the error.
     */
    public function socialAccount()
    {
        return $this->belongsTo(SocialAccount::class);
    }

    /**
     * Get a formatted error message.
     *
     * @return string
     */
    public function getFormattedMessage()
    {
        $message = $this->error_text;
        
        if ($this->response_code) {
            $message = "[{$this->response_code}] {$message}";
        }
        
        if ($this->socialAccount) {
            $message .= " (Provider: {$this->socialAccount->provider}, Account: {$this->socialAccount->provider_user_id})";
        }
        
        return $message;
    }

    /**
     * Check if the error is retryable.
     *
     * @return bool
     */
    public function isRetryable()
    {
        // Common retryable HTTP status codes
        $retryableCodes = [429, 500, 502, 503, 504];
        
        // If there's no response code, assume it's retryable
        if (!$this->response_code) {
            return true;
        }
        
        return in_array($this->response_code, $retryableCodes);
    }
}
