<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ReputationRecommendation extends Model
{
    use HasFactory;

    protected $fillable = [
        'reputation_score_id',
        'brand_id',
        'title',
        'description',
        'priority',
        'category',
        'estimated_impact',
        'effort_level',
        'estimated_days',
        'status',
        'action_taken',
        'assigned_to',
        'completed_at',
        'ai_reasoning',
        'supporting_data',
    ];

    protected $casts = [
        'estimated_impact' => 'decimal:2',
        'estimated_days' => 'integer',
        'completed_at' => 'datetime',
        'supporting_data' => 'array',
    ];

    /**
     * Get the reputation score that owns the recommendation.
     */
    public function reputationScore(): BelongsTo
    {
        return $this->belongsTo(ReputationScore::class);
    }

    /**
     * Get the brand that owns the recommendation.
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the user assigned to this recommendation.
     */
    public function assignedUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    /**
     * Mark recommendation as completed.
     */
    public function markAsCompleted(string $actionTaken = null): void
    {
        $this->update([
            'status' => 'completed',
            'action_taken' => $actionTaken,
            'completed_at' => now(),
        ]);
    }

    /**
     * Mark recommendation as in progress.
     */
    public function markAsInProgress(): void
    {
        $this->update(['status' => 'in_progress']);
    }

    /**
     * Dismiss recommendation.
     */
    public function dismiss(): void
    {
        $this->update(['status' => 'dismissed']);
    }

    /**
     * Assign to user.
     */
    public function assignTo(User $user): void
    {
        $this->update(['assigned_to' => $user->id]);
    }

    /**
     * Get the ROI score (impact vs effort).
     */
    public function getRoiScoreAttribute(): float
    {
        $effortWeight = [
            'low' => 1,
            'medium' => 2,
            'high' => 3,
        ];

        $effort = $effortWeight[$this->effort_level] ?? 2;
        return round($this->estimated_impact / $effort, 2);
    }

    /**
     * Check if recommendation is actionable.
     */
    public function isActionable(): bool
    {
        return in_array($this->status, ['pending', 'in_progress']);
    }

    /**
     * Check if recommendation is completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Scope to get pending recommendations.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope to get in progress recommendations.
     */
    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    /**
     * Scope to get completed recommendations.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope to get high priority recommendations.
     */
    public function scopeHighPriority($query)
    {
        return $query->where('priority', 'high');
    }

    /**
     * Scope to order by priority.
     */
    public function scopeOrderByPriority($query)
    {
        return $query->orderByRaw("FIELD(priority, 'high', 'medium', 'low')");
    }

    /**
     * Scope to order by ROI.
     */
    public function scopeOrderByRoi($query)
    {
        return $query->orderByRaw('estimated_impact / CASE effort_level WHEN "low" THEN 1 WHEN "medium" THEN 2 WHEN "high" THEN 3 END DESC');
    }

    /**
     * Scope to get recommendations by category.
     */
    public function scopeByCategory($query, string $category)
    {
        return $query->where('category', $category);
    }
}