<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Plan extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'type',
        'price',
        'currency',
        'interval',
        'is_active',
        'trial_days',
        'features',
        'max_social_accounts',
        'max_campaigns',
        'max_proposals',
        'max_content_ideas_per_month',
        'max_reels_per_month',
        'max_ad_variants',
        'has_ads_manager',
        'has_analytics',
        'has_competitor_insights',
        'has_localization',
        'has_messaging',
        'has_api_access',
        'has_white_label',
        'has_priority_support',
        'sort_order',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean',
        'trial_days' => 'integer',
        'features' => 'array',
        'max_social_accounts' => 'integer',
        'max_campaigns' => 'integer',
        'max_proposals' => 'integer',
        'max_content_ideas_per_month' => 'integer',
        'max_reels_per_month' => 'integer',
        'max_ad_variants' => 'integer',
        'has_ads_manager' => 'boolean',
        'has_analytics' => 'boolean',
        'has_competitor_insights' => 'boolean',
        'has_localization' => 'boolean',
        'has_messaging' => 'boolean',
        'has_api_access' => 'boolean',
        'has_white_label' => 'boolean',
        'has_priority_support' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the subscriptions for this plan.
     */
    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Check if the plan is free.
     */
    public function isFree(): bool
    {
        return $this->type === 'free' || $this->price == 0;
    }

    /**
     * Check if the plan has a trial period.
     */
    public function hasTrial(): bool
    {
        return $this->trial_days > 0;
    }

    /**
     * Check if a feature is unlimited.
     */
    public function isUnlimited(string $feature): bool
    {
        $value = $this->getAttribute($feature);
        return $value === -1;
    }

    /**
     * Get the formatted price.
     */
    public function getFormattedPriceAttribute(): string
    {
        if ($this->isFree()) {
            return 'Free';
        }

        $symbol = $this->currency === 'INR' ? '₹' : '$';
        return $symbol . number_format($this->price, 2);
    }

    /**
     * Get the price per interval text.
     */
    public function getPriceTextAttribute(): string
    {
        if ($this->isFree()) {
            return 'Free forever';
        }

        if ($this->slug === 'enterprise') {
            return 'Custom pricing';
        }

        return $this->formatted_price . '/' . $this->interval;
    }

    /**
     * Scope to get only active plans.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to order by sort order.
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order');
    }

    /**
     * Check if user can access a feature based on usage.
     */
    public function canUseFeature(string $feature, int $currentUsage): bool
    {
        $limit = $this->getAttribute($feature);
        
        // -1 means unlimited
        if ($limit === -1) {
            return true;
        }
        
        return $currentUsage < $limit;
    }
}