<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CampaignFinancial extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'campaign_id',
        'brand_id',
        'ad_spend',
        'influencer_payments',
        'production_costs',
        'other_costs',
        'total_spend',
        'clicks',
        'impressions',
        'conversions',
        'avg_order_value',
        'total_revenue',
        'roi_percentage',
        'breakeven_days',
        'cost_per_click',
        'cost_per_conversion',
        'conversion_rate',
        'period_start',
        'period_end',
        'spend_breakdown',
        'performance_metrics',
    ];

    protected $casts = [
        'ad_spend' => 'decimal:2',
        'influencer_payments' => 'decimal:2',
        'production_costs' => 'decimal:2',
        'other_costs' => 'decimal:2',
        'total_spend' => 'decimal:2',
        'clicks' => 'integer',
        'impressions' => 'integer',
        'conversions' => 'integer',
        'avg_order_value' => 'decimal:2',
        'total_revenue' => 'decimal:2',
        'roi_percentage' => 'decimal:2',
        'breakeven_days' => 'integer',
        'cost_per_click' => 'decimal:2',
        'cost_per_conversion' => 'decimal:2',
        'conversion_rate' => 'decimal:2',
        'period_start' => 'date',
        'period_end' => 'date',
        'spend_breakdown' => 'array',
        'performance_metrics' => 'array',
    ];

    /**
     * Boot method to auto-calculate metrics
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($financial) {
            $financial->calculateMetrics();
        });
    }

    /**
     * Calculate all financial metrics
     */
    public function calculateMetrics(): void
    {
        // Calculate total spend
        $this->total_spend = $this->ad_spend + $this->influencer_payments + 
                            $this->production_costs + $this->other_costs;

        // Calculate ROI percentage
        if ($this->total_spend > 0) {
            $this->roi_percentage = (($this->total_revenue - $this->total_spend) / $this->total_spend) * 100;
        }

        // Calculate cost per click
        if ($this->clicks > 0) {
            $this->cost_per_click = $this->total_spend / $this->clicks;
        }

        // Calculate cost per conversion
        if ($this->conversions > 0) {
            $this->cost_per_conversion = $this->total_spend / $this->conversions;
        }

        // Calculate conversion rate
        if ($this->clicks > 0) {
            $this->conversion_rate = ($this->conversions / $this->clicks) * 100;
        }

        // Calculate breakeven days (simplified estimation)
        if ($this->total_revenue > 0 && $this->period_start && $this->period_end) {
            $periodDays = $this->period_start->diffInDays($this->period_end);
            $dailyRevenue = $this->total_revenue / max($periodDays, 1);
            
            if ($dailyRevenue > 0) {
                $this->breakeven_days = (int) ceil($this->total_spend / $dailyRevenue);
            }
        }
    }

    /**
     * Get the campaign that owns the financial data
     */
    public function campaign(): BelongsTo
    {
        return $this->belongsTo(Campaign::class);
    }

    /**
     * Get the brand that owns the financial data
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(User::class, 'brand_id');
    }

    /**
     * Scope to filter by date range
     */
    public function scopeInPeriod($query, $startDate, $endDate)
    {
        return $query->where(function ($q) use ($startDate, $endDate) {
            $q->whereBetween('period_start', [$startDate, $endDate])
              ->orWhereBetween('period_end', [$startDate, $endDate])
              ->orWhere(function ($q2) use ($startDate, $endDate) {
                  $q2->where('period_start', '<=', $startDate)
                     ->where('period_end', '>=', $endDate);
              });
        });
    }

    /**
     * Scope to filter by brand
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Get profit/loss amount
     */
    public function getProfitAttribute(): float
    {
        return $this->total_revenue - $this->total_spend;
    }

    /**
     * Check if campaign is profitable
     */
    public function isProfitable(): bool
    {
        return $this->total_revenue > $this->total_spend;
    }

    /**
     * Get ROI status (positive, negative, breakeven)
     */
    public function getRoiStatus(): string
    {
        if ($this->roi_percentage === null) {
            return 'unknown';
        }

        if ($this->roi_percentage > 5) {
            return 'positive';
        } elseif ($this->roi_percentage < -5) {
            return 'negative';
        }

        return 'breakeven';
    }
}