<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AdVariant extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'ad_id',
        'headline',
        'body',
        'cta',
        'predicted_metrics',
        'is_active',
        'is_winner',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'predicted_metrics' => 'array',
        'is_active' => 'boolean',
        'is_winner' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the ad that owns the variant.
     */
    public function ad()
    {
        return $this->belongsTo(Ad::class);
    }

    /**
     * Get the performance metrics for the variant.
     */
    public function performanceMetrics()
    {
        return $this->hasMany(AdPerformanceMetric::class);
    }

    /**
     * Get the latest performance metrics for the variant.
     */
    public function latestPerformanceMetrics()
    {
        return $this->performanceMetrics()->latest();
    }

    /**
     * Get the performance metrics for a specific platform.
     */
    public function performanceMetricsForPlatform($platform)
    {
        return $this->performanceMetrics()->where('platform', $platform)->first();
    }

    /**
     * Calculate the click-through rate (CTR) for this variant.
     */
    public function calculateCTR()
    {
        $metrics = $this->latestPerformanceMetrics()->first();
        if (!$metrics || $metrics->impressions == 0) {
            return 0;
        }
        return ($metrics->clicks / $metrics->impressions) * 100;
    }

    /**
     * Calculate the conversion rate for this variant.
     */
    public function calculateConversionRate()
    {
        $metrics = $this->latestPerformanceMetrics()->first();
        if (!$metrics || $metrics->clicks == 0) {
            return 0;
        }
        return ($metrics->conversions / $metrics->clicks) * 100;
    }
}