<?php

namespace App\Jobs;

use App\Models\ReelRequest;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SendReelToProcessor implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $reelRequest;

    /**
     * Create a new job instance.
     */
    public function __construct(ReelRequest $reelRequest)
    {
        $this->reelRequest = $reelRequest;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            // Prepare inputs with signed URLs
            $inputs = [];
            foreach ($this->reelRequest->inputs as $input) {
                $inputs[] = [
                    'id' => $input->id,
                    'source_url' => $this->getSignedUrl($input),
                    'provider' => $input->provider,
                    'meta' => $input->meta
                ];
            }

            // Generate a unique job ID
            $jobId = Str::uuid()->toString();

            // Prepare the payload
            $payload = [
                'request_id' => $this->reelRequest->id,
                'job_id' => $jobId,
                'inputs' => $inputs,
                'template' => $this->reelRequest->template,
                'options' => $this->reelRequest->options,
                'callback_url' => route('api.reels.callback')
            ];

            // Compute HMAC signature
            $signature = hash_hmac('sha256', json_encode($payload), config('reels.processor_secret'));

            // Send request to processor
            $response = Http::withHeaders([
                'X-Processor-Signature' => $signature,
                'Content-Type' => 'application/json'
            ])->post(config('reels.processor_url'), $payload);

            // Log the request for debugging
            Log::info('Reel processor request sent', [
                'request_id' => $this->reelRequest->id,
                'job_id' => $jobId,
                'url' => config('reels.processor_url'),
                'payload' => $payload,
                'response_status' => $response->status(),
                'response_body' => $response->body()
            ]);

            if ($response->successful() || $response->status() == 202) {
                // Update the reel request with job ID and processing status
                $this->reelRequest->update([
                    'job_id' => $jobId,
                    'status' => 'processing'
                ]);
            } else {
                // Mark as failed
                $this->reelRequest->markFailed('Processor returned status: ' . $response->status());
            }
        } catch (\Exception $e) {
            Log::error('Failed to send reel to processor: ' . $e->getMessage(), [
                'request_id' => $this->reelRequest->id,
                'trace' => $e->getTraceAsString()
            ]);

            // Mark as failed
            $this->reelRequest->markFailed('Failed to send to processor: ' . $e->getMessage());
        }
    }

    /**
     * Get signed URL for the input source.
     */
    protected function getSignedUrl($input)
    {
        // If it's a social post with media URL, return it directly
        if ($input->sourcePost && $input->sourcePost->media_url) {
            return $input->sourcePost->media_url;
        }

        // If it's a direct source URL, return it
        if ($input->source_url) {
            return $input->source_url;
        }

        // If we have a local file, create a signed URL
        if ($input->sourcePost && $input->sourcePost->media_path) {
            $disk = config('reels.storage_disk', 'public');
            $expireMinutes = config('reels.signed_url_expire', 300);
            
            if (Storage::disk($disk)->exists($input->sourcePost->media_path)) {
                return Storage::disk($disk)->temporaryUrl(
                    $input->sourcePost->media_path,
                    now()->addMinutes($expireMinutes)
                );
            }
        }

        return null;
    }
}