<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class UpdateAdRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        // Authorization is handled by the policy
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, mixed>
     */
    public function rules()
    {
        $brand = auth()->user()->brand;
        
        return [
            'campaign_id' => [
                'required',
                'exists:campaigns,id,brand_id,' . ($brand ? $brand->id : 0),
            ],
            'name' => [
                'required',
                'string',
                'max:255',
            ],
            'product_info' => [
                'required',
                'string',
                'max:2000',
            ],
            'audience' => [
                'required',
                'string',
                'max:1000',
            ],
            'tone' => [
                'required',
                'string',
                'in:professional,casual,enthusiastic,authoritative,humorous,empathetic,urgent',
            ],
            'budget' => [
                'nullable',
                'numeric',
                'min:0',
                'max:999999999.99',
            ],
            'currency' => [
                'nullable',
                'string',
                'size:3',
                'in:USD,EUR,GBP,INR,CAD,AUD,JPY,CNY',
            ],
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages()
    {
        return [
            'campaign_id.required' => 'Please select a campaign for this ad.',
            'campaign_id.exists' => 'The selected campaign is invalid or does not belong to your brand.',
            'name.required' => 'Please provide a name for your ad.',
            'name.max' => 'The ad name cannot exceed 255 characters.',
            'product_info.required' => 'Please provide information about your product or service.',
            'product_info.max' => 'The product information cannot exceed 2000 characters.',
            'audience.required' => 'Please describe your target audience.',
            'audience.max' => 'The audience description cannot exceed 1000 characters.',
            'tone.required' => 'Please select a tone for your ad.',
            'tone.in' => 'The selected tone is invalid.',
            'budget.numeric' => 'The budget must be a valid number.',
            'budget.min' => 'The budget cannot be negative.',
            'budget.max' => 'The budget is too large.',
            'currency.size' => 'The currency code must be exactly 3 characters.',
            'currency.in' => 'The selected currency is not supported.',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes()
    {
        return [
            'campaign_id' => 'campaign',
            'product_info' => 'product information',
        ];
    }
}