<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\AiJob;

class ContentIdeaRateLimit
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $user = Auth::user();
        
        if (!$user) {
            return $next($request);
        }
        
        // Check if user is a brand admin or creator
        if ($user->hasRole('brand_admin')) {
            // Apply rate limiting for brand admins
            // Check daily limit for content idea generation
            $dailyLimit = config('content_ideas.generation_rate_limit_per_day', 5);
            $todayCount = AiJob::where('brand_id', $user->brand_id)
                ->where('type', 'content_idea_generation')
                ->whereDate('created_at', today())
                ->count();
                
            if ($todayCount >= $dailyLimit) {
                return redirect()->back()->with('error', 'You have reached your daily content idea generation limit of ' . $dailyLimit . ' requests.');
            }
            
            // Check concurrent limit for content idea generation
            $concurrentLimit = config('content_ideas.max_concurrent_per_brand', 2);
            $concurrentCount = AiJob::where('brand_id', $user->brand_id)
                ->where('type', 'content_idea_generation')
                ->whereIn('status', ['queued', 'generating'])
                ->count();
                
            if ($concurrentCount >= $concurrentLimit) {
                return redirect()->back()->with('error', 'You have reached your concurrent content idea generation limit of ' . $concurrentLimit . ' requests.');
            }
        } elseif ($user->hasRole('creator')) {
            // Apply rate limiting for creators
            // Check daily limit for content idea generation
            $dailyLimit = config('content_ideas.creator_generation_rate_limit_per_day', 10);
            $todayCount = AiJob::where('creator_id', $user->id)
                ->where('type', 'content_idea_generation')
                ->whereDate('created_at', today())
                ->count();
                
            if ($todayCount >= $dailyLimit) {
                return redirect()->back()->with('error', 'You have reached your daily content idea generation limit of ' . $dailyLimit . ' requests.');
            }
            
            // Check concurrent limit for content idea generation
            $concurrentLimit = config('content_ideas.creator_max_concurrent', 3);
            $concurrentCount = AiJob::where('creator_id', $user->id)
                ->where('type', 'content_idea_generation')
                ->whereIn('status', ['queued', 'generating'])
                ->count();
                
            if ($concurrentCount >= $concurrentLimit) {
                return redirect()->back()->with('error', 'You have reached your concurrent content idea generation limit of ' . $concurrentLimit . ' requests.');
            }
        } else {
            // For other user types, don't apply rate limiting
            return $next($request);
        }
        
        return $next($request);
    }
}