<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PlanController extends Controller
{
    /**
     * Display a listing of subscription plans.
     */
    public function index()
    {
        $plans = Plan::orderBy('sort_order')->get();
        
        return view('superadmin.plans.index', compact('plans'));
    }

    /**
     * Show the form for creating a new plan.
     */
    public function create()
    {
        return view('superadmin.plans.create');
    }

    /**
     * Store a newly created plan in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:plans,slug',
            'description' => 'nullable|string',
            'type' => 'required|in:free,paid',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:3',
            'interval' => 'required|in:monthly,yearly',
            'trial_days' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'features' => 'nullable|array',
            'max_social_accounts' => 'required|integer|min:-1',
            'max_campaigns' => 'required|integer|min:-1',
            'max_proposals' => 'required|integer|min:-1',
            'max_content_ideas_per_month' => 'required|integer|min:-1',
            'max_reels_per_month' => 'required|integer|min:-1',
            'max_ad_variants' => 'required|integer|min:-1',
            'has_ads_manager' => 'boolean',
            'has_analytics' => 'boolean',
            'has_competitor_insights' => 'boolean',
            'has_localization' => 'boolean',
            'has_messaging' => 'boolean',
            'has_api_access' => 'boolean',
            'has_white_label' => 'boolean',
            'has_priority_support' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Set default sort order
        if (!isset($validated['sort_order'])) {
            $validated['sort_order'] = Plan::max('sort_order') + 1;
        }

        // Convert features array to JSON if provided
        if (isset($validated['features']) && is_array($validated['features'])) {
            $validated['features'] = array_filter($validated['features']);
        }

        $plan = Plan::create($validated);

        return redirect()->route('superadmin.plans.index')
            ->with('success', 'Plan created successfully.');
    }

    /**
     * Display the specified plan.
     */
    public function show(Plan $plan)
    {
        $subscriptionsCount = $plan->subscriptions()->count();
        $activeSubscriptionsCount = $plan->subscriptions()->active()->count();
        $monthlyRevenue = $plan->subscriptions()->active()->count() * $plan->price;

        return view('superadmin.plans.show', compact(
            'plan',
            'subscriptionsCount',
            'activeSubscriptionsCount',
            'monthlyRevenue'
        ));
    }

    /**
     * Show the form for editing the specified plan.
     */
    public function edit(Plan $plan)
    {
        return view('superadmin.plans.edit', compact('plan'));
    }

    /**
     * Update the specified plan in storage.
     */
    public function update(Request $request, Plan $plan)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:plans,slug,' . $plan->id,
            'description' => 'nullable|string',
            'type' => 'required|in:free,paid',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:3',
            'interval' => 'required|in:monthly,yearly',
            'trial_days' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'features' => 'nullable|array',
            'max_social_accounts' => 'required|integer|min:-1',
            'max_campaigns' => 'required|integer|min:-1',
            'max_proposals' => 'required|integer|min:-1',
            'max_content_ideas_per_month' => 'required|integer|min:-1',
            'max_reels_per_month' => 'required|integer|min:-1',
            'max_ad_variants' => 'required|integer|min:-1',
            'has_ads_manager' => 'boolean',
            'has_analytics' => 'boolean',
            'has_competitor_insights' => 'boolean',
            'has_localization' => 'boolean',
            'has_messaging' => 'boolean',
            'has_api_access' => 'boolean',
            'has_white_label' => 'boolean',
            'has_priority_support' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        // Convert features array to JSON if provided
        if (isset($validated['features']) && is_array($validated['features'])) {
            $validated['features'] = array_filter($validated['features']);
        }

        $plan->update($validated);

        return redirect()->route('superadmin.plans.index')
            ->with('success', 'Plan updated successfully.');
    }

    /**
     * Remove the specified plan from storage.
     */
    public function destroy(Plan $plan)
    {
        // Check if plan has active subscriptions
        if ($plan->subscriptions()->active()->count() > 0) {
            return redirect()->route('superadmin.plans.index')
                ->with('error', 'Cannot delete plan with active subscriptions.');
        }

        $plan->delete();

        return redirect()->route('superadmin.plans.index')
            ->with('success', 'Plan deleted successfully.');
    }

    /**
     * Toggle plan active status.
     */
    public function toggleStatus(Plan $plan)
    {
        $plan->update(['is_active' => !$plan->is_active]);

        $status = $plan->is_active ? 'activated' : 'deactivated';

        return redirect()->route('superadmin.plans.index')
            ->with('success', "Plan {$status} successfully.");
    }
}