<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class AgencyController extends Controller
{
    /**
     * Display a listing of agencies
     */
    public function index()
    {
        $agencies = User::role('agency')
            ->with(['brands', 'tenant'])
            ->withCount(['brands'])
            ->latest()
            ->paginate(20);

        return view('superadmin.agencies.index', compact('agencies'));
    }

    /**
     * Show the form for creating a new agency
     */
    public function create()
    {
        return view('superadmin.agencies.create');
    }

    /**
     * Store a newly created agency
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'company_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
        ]);

        $agency = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => bcrypt($validated['password']),
            'phone' => $validated['phone'] ?? null,
            'meta' => [
                'company_name' => $validated['company_name'] ?? null,
            ],
        ]);

        $agency->assignRole('agency');

        return redirect()->route('superadmin.agencies.index')
            ->with('success', 'Agency created successfully.');
    }

    /**
     * Display the specified agency
     */
    public function show(User $agency)
    {
        $agency->load(['brands', 'tenant']);
        
        return view('superadmin.agencies.show', compact('agency'));
    }

    /**
     * Show the form for editing the specified agency
     */
    public function edit(User $agency)
    {
        return view('superadmin.agencies.edit', compact('agency'));
    }

    /**
     * Update the specified agency
     */
    public function update(Request $request, User $agency)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $agency->id,
            'company_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
        ]);

        $agency->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'meta' => array_merge($agency->meta ?? [], [
                'company_name' => $validated['company_name'] ?? null,
            ]),
        ]);

        return redirect()->route('superadmin.agencies.index')
            ->with('success', 'Agency updated successfully.');
    }

    /**
     * Remove the specified agency
     */
    public function destroy(User $agency)
    {
        $agency->delete();

        return redirect()->route('superadmin.agencies.index')
            ->with('success', 'Agency deleted successfully.');
    }
}
