<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\CreatorScheduledPost;
use App\Models\SocialAccount;
use App\Models\Campaign;
use App\Models\AutoReplyTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Http\Requests\StoreCreatorScheduledPostRequest;
use App\Http\Requests\UpdateCreatorScheduledPostRequest;

class ScheduledPostController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = CreatorScheduledPost::where('user_id', auth()->id())
            ->with(['socialAccount', 'campaign']);

        // Apply filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('campaign_id')) {
            $query->where('campaign_id', $request->campaign_id);
        }

        // Get scheduled posts
        $scheduledPosts = $query->latest()->paginate(20);

        // Get campaigns for filter dropdown
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();

        return view('creator.scheduled-posts.index', compact('scheduledPosts', 'campaigns'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get social accounts for the creator
        $socialAccounts = SocialAccount::where('user_id', auth()->id())
            ->where('tenant_id', tenant('id'))
            ->get();

        // Get campaigns for the creator
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();
        
        // Get auto reply templates
        $autoReplyTemplates = AutoReplyTemplate::where('user_id', auth()->id())->get();

        return view('creator.scheduled-posts.create', compact('socialAccounts', 'campaigns', 'autoReplyTemplates'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCreatorScheduledPostRequest $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                // Create the scheduled post
                $scheduledPost = CreatorScheduledPost::create([
                    'user_id' => auth()->id(),
                    'social_account_id' => $request->social_account_id,
                    'caption' => $request->caption,
                    'media_refs' => $request->media_refs,
                    'publish_time' => $request->publish_time,
                    'timezone' => $request->timezone,
                    'status' => 'scheduled',
                    'retry_policy' => [
                        'max_attempts' => 3,
                        'backoff' => 'exponential',
                        'base_delay' => 60,
                        'max_delay' => 86400,
                        'multiplier' => 2
                    ],
                    'idempotency_key' => Str::uuid(),
                    'campaign_id' => $request->campaign_id,
                    'post_visibility' => $request->post_visibility,
                    'recurrence' => $request->recurrence,
                    'auto_reply_enabled' => $request->auto_reply_enabled ?? false,
                    'auto_reply_template_id' => $request->auto_reply_template_id,
                ]);

                return redirect()->route('creator.scheduled-posts.show', $scheduledPost)
                    ->with('success', 'Post scheduled successfully.');
            });
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to schedule post: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only view their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        $scheduledPost->load(['socialAccount', 'campaign', 'autoReplyTemplate']);

        return view('creator.scheduled-posts.show', compact('scheduledPost'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only edit their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        // Get social accounts for the creator
        $socialAccounts = SocialAccount::where('user_id', auth()->id())
            ->where('tenant_id', tenant('id'))
            ->get();

        // Get campaigns for the creator
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();
        
        // Get auto reply templates
        $autoReplyTemplates = AutoReplyTemplate::where('user_id', auth()->id())->get();

        $scheduledPost->load(['socialAccount', 'campaign']);

        return view('creator.scheduled-posts.edit', compact('scheduledPost', 'socialAccounts', 'campaigns', 'autoReplyTemplates'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCreatorScheduledPostRequest $request, CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only update their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        try {
            $scheduledPost->update([
                'social_account_id' => $request->social_account_id,
                'caption' => $request->caption,
                'media_refs' => $request->media_refs,
                'publish_time' => $request->publish_time,
                'timezone' => $request->timezone,
                'campaign_id' => $request->campaign_id,
                'post_visibility' => $request->post_visibility,
                'recurrence' => $request->recurrence,
                'auto_reply_enabled' => $request->auto_reply_enabled ?? false,
                'auto_reply_template_id' => $request->auto_reply_template_id,
            ]);

            return redirect()->route('creator.scheduled-posts.show', $scheduledPost)
                ->with('success', 'Post updated successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update post: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only delete their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        try {
            $scheduledPost->delete();

            return redirect()->route('creator.scheduled-posts.index')
                ->with('success', 'Post deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete post: ' . $e->getMessage());
        }
    }

    /**
     * Publish a scheduled post immediately.
     */
    public function publishNow(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only publish their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        // Check if post is in a valid status for immediate publishing
        if (!in_array($scheduledPost->status, ['scheduled', 'draft'])) {
            return back()->with('error', 'Post cannot be published in its current status.');
        }

        try {
            // Dispatch job to process the scheduled post immediately
            \App\Jobs\ProcessScheduledPost::dispatch($scheduledPost);
            
            // Mark as publishing
            $scheduledPost->markPublishing();

            return back()->with('success', 'Post is being published now.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to publish post: ' . $e->getMessage());
        }
    }

    /**
     * Reschedule a post.
     */
    public function reschedule(CreatorScheduledPost $scheduledPost, Request $request)
    {
        // Ensure the user can only reschedule their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        $request->validate([
            'publish_time' => 'required|date',
            'timezone' => 'required|string',
        ]);

        try {
            $scheduledPost->update([
                'publish_time' => $request->publish_time,
                'timezone' => $request->timezone,
                'status' => 'scheduled',
            ]);

            return back()->with('success', 'Post has been rescheduled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to reschedule post: ' . $e->getMessage());
        }
    }

    /**
     * Cancel a scheduled post.
     */
    public function cancel(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only cancel their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        try {
            $scheduledPost->markCancelled();

            return back()->with('success', 'Post has been cancelled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel post: ' . $e->getMessage());
        }
    }

    /**
     * Duplicate a scheduled post.
     */
    public function duplicate(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only duplicate their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        try {
            // Create a duplicate of the scheduled post
            $duplicatePost = $scheduledPost->replicate();
            $duplicatePost->status = 'draft';
            $duplicatePost->remote_post_id = null;
            $duplicatePost->provider_response = null;
            $duplicatePost->error_message = null;
            $duplicatePost->attempt_count = 0;
            $duplicatePost->last_attempt_at = null;
            $duplicatePost->next_retry_at = null;
            $duplicatePost->idempotency_key = Str::uuid();
            $duplicatePost->save();

            return redirect()->route('creator.scheduled-posts.edit', $duplicatePost)
                ->with('success', 'Post duplicated successfully. You can now edit the duplicated post.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to duplicate post: ' . $e->getMessage());
        }
    }

    /**
     * Retry processing a failed post.
     */
    public function retry(CreatorScheduledPost $scheduledPost)
    {
        // Ensure the user can only retry their own scheduled posts
        if ($scheduledPost->user_id != auth()->id()) {
            abort(403);
        }

        if ($scheduledPost->status != 'failed') {
            return back()->with('error', 'Only failed posts can be retried.');
        }

        try {
            // Reset attempt count and error message
            $scheduledPost->update([
                'status' => 'scheduled',
                'error_message' => null,
                'attempt_count' => 0,
                'last_attempt_at' => null,
                'next_retry_at' => null,
            ]);

            return back()->with('success', 'Post processing has been restarted.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to retry post processing: ' . $e->getMessage());
        }
    }
    
    /**
     * Bulk publish scheduled posts.
     */
    public function bulkPublish(Request $request)
    {
        $request->validate([
            'post_ids' => 'required|array',
            'post_ids.*' => 'exists:creator_scheduled_posts,id',
        ]);
        
        $postIds = $request->post_ids;
        
        // Get the posts that belong to the current user
        $scheduledPosts = CreatorScheduledPost::where('user_id', auth()->id())
            ->whereIn('id', $postIds)
            ->whereIn('status', ['scheduled', 'draft'])
            ->get();
        
        $publishedCount = 0;
        $failedCount = 0;
        
        foreach ($scheduledPosts as $scheduledPost) {
            try {
                // Dispatch job to process the scheduled post immediately
                \App\Jobs\ProcessScheduledPost::dispatch($scheduledPost);
                
                // Mark as publishing
                $scheduledPost->markPublishing();
                
                $publishedCount++;
            } catch (\Exception $e) {
                $failedCount++;
                // Log the error but continue with other posts
                \Log::error('Failed to publish scheduled post: ' . $e->getMessage(), [
                    'scheduled_post_id' => $scheduledPost->id,
                ]);
            }
        }
        
        if ($publishedCount > 0 && $failedCount == 0) {
            return back()->with('success', "Successfully published {$publishedCount} posts.");
        } elseif ($publishedCount > 0 && $failedCount > 0) {
            return back()->with('warning', "Published {$publishedCount} posts, but {$failedCount} failed to publish.");
        } else {
            return back()->with('error', "Failed to publish any posts.");
        }
    }
}