<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\SocialAggregate;
use App\Models\SocialPost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class AnalyticsController extends Controller
{
    /**
     * Display the analytics dashboard
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // Get date range from request or default to 30 days
        $days = $request->input('days', 30);
        $startDate = now()->subDays($days);
        
        // Get the creator's social accounts
        $socialAccounts = auth()->user()->socialAccounts;
        
        // Get analytics data with default parameters
        $analyticsData = $this->getAnalyticsData($days);
        
        // Extract KPIs
        $totalFollowers = $analyticsData['total_followers'];
        $totalPosts = $analyticsData['total_posts'];
        $engagementRate = $analyticsData['engagement_rate'];
        $totalLikes = $analyticsData['total_likes'];
        $totalComments = $analyticsData['total_comments'];
        $totalShares = $analyticsData['total_shares'];
        $totalSaves = $analyticsData['total_saves'];
        $totalImpressions = $analyticsData['total_impressions'];
        $followerGrowth = $analyticsData['follower_growth'];
        $topContentType = $analyticsData['top_content_type'];
        $bestPostingTime = $analyticsData['best_posting_time'];
        $topHashtags = $analyticsData['top_hashtags'];
        $personalBestPosts = $analyticsData['personal_best_posts'];
            
        return view('creator.analytics.index', compact(
            'socialAccounts',
            'totalFollowers',
            'totalPosts',
            'engagementRate',
            'totalLikes',
            'totalComments',
            'totalShares',
            'totalSaves',
            'totalImpressions',
            'followerGrowth',
            'topContentType',
            'bestPostingTime',
            'topHashtags',
            'personalBestPosts',
            'days'
        ));
    }
    
    /**
     * Return analytics data as JSON for charts
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function data(Request $request)
    {
        $provider = $request->input('provider');
        $days = $request->input('days', 30);
        
        // Get views data for the last N days
        $viewsData = SocialPost::whereHas('socialAccount', function ($query) use ($provider) {
            $query->where('user_id', auth()->id());
            if ($provider) {
                $query->where('provider', $provider);
            }
        })->where('published_at', '>=', now()->subDays($days))
        ->selectRaw('DATE(published_at) as date, SUM(JSON_EXTRACT(metrics, "$.views")) as views')
        ->groupBy('date')
        ->orderBy('date')
        ->get();
        
        // Format data for Chart.js
        $labels = [];
        $data = [];
        
        // Fill in missing dates
        $startDate = now()->subDays($days);
        for ($i = 0; $i <= $days; $i++) {
            $date = $startDate->copy()->addDays($i)->format('Y-m-d');
            $labels[] = $date;
            $data[] = 0;
        }
        
        // Fill in actual data
        foreach ($viewsData as $item) {
            $index = array_search($item->date, $labels);
            if ($index !== false) {
                $data[$index] = (int) $item->views;
            }
        }
        
        return response()->json([
            'labels' => $labels,
            'data' => $data
        ]);
    }
    
    /**
     * Get comprehensive analytics data
     *
     * @param int $days
     * @return array
     */
    protected function getAnalyticsData($days = 30)
    {
        $userId = auth()->id();
        $cacheKey = "creator_analytics_{$userId}_{$days}";
        
        return Cache::remember($cacheKey, 300, function () use ($userId, $days) {
            // Get date range
            $startDate = now()->subDays($days);
            
            // Build query for social posts
            $postsQuery = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
                $query->where('user_id', $userId);
            })->where('published_at', '>=', $startDate);
            
            // Get all posts for calculations
            $posts = $postsQuery->get();
            
            // Calculate metrics
            $totalImpressions = 0;
            $totalLikes = 0;
            $totalComments = 0;
            $totalShares = 0;
            $totalSaves = 0;
            
            foreach ($posts as $post) {
                $metrics = $post->metrics ?? [];
                $totalImpressions += $metrics['views'] ?? 0;
                $totalLikes += $metrics['likes'] ?? 0;
                $totalComments += $metrics['comments'] ?? 0;
                $totalShares += $metrics['shares'] ?? 0;
                $totalSaves += $metrics['saves'] ?? 0;
            }
            
            // Calculate engagement rate: (likes + comments + shares + saves) / impressions
            $engagementRate = 0;
            if ($totalImpressions > 0) {
                $engagementRate = round((($totalLikes + $totalComments + $totalShares + $totalSaves) / $totalImpressions) * 100, 2);
            }
            
            // Get follower data
            $totalFollowers = SocialAggregate::where('user_id', $userId)
                ->where('key', 'total_followers')
                ->value('value');
                
            $totalFollowers = $totalFollowers ? $totalFollowers['value'] ?? 0 : 0;
            
            // Get follower growth (compare with previous period)
            $previousStartDate = now()->subDays($days * 2);
            $previousEndDate = now()->subDays($days);
            
            $previousFollowers = SocialAggregate::where('user_id', $userId)
                ->where('key', 'total_followers')
                ->where('computed_at', '<=', $previousEndDate)
                ->orderBy('computed_at', 'desc')
                ->first();
                
            $previousFollowersCount = $previousFollowers ? $previousFollowers->getNumericValue() : 0;
            $followerGrowth = 0;
            
            if ($previousFollowersCount > 0) {
                $followerGrowth = round((($totalFollowers - $previousFollowersCount) / $previousFollowersCount) * 100, 2);
            }
            
            // Get total posts
            $totalPosts = $posts->count();
            
            // Get time series data for charts
            $timeSeriesData = $this->getTimeSeriesData($postsQuery, $days);
            
            // Get top content type
            $topContentType = $this->getTopContentType($posts);
            
            // Get best posting time
            $bestPostingTime = $this->getBestPostingTime($posts);
            
            // Get top hashtags (simplified implementation)
            $topHashtags = $this->getTopHashtags($posts);
            
            // Get personal best posts
            $personalBestPosts = $posts->sortByDesc(function ($post) {
                $metrics = $post->metrics ?? [];
                return ($metrics['likes'] ?? 0) + ($metrics['comments'] ?? 0) + ($metrics['shares'] ?? 0);
            })->take(5);
            
            $personalBestPostsData = [];
            foreach ($personalBestPosts as $post) {
                $metrics = $post->metrics ?? [];
                $personalBestPostsData[] = [
                    'title' => $post->title ?? 'Untitled',
                    'provider' => $post->socialAccount->provider ?? 'Unknown',
                    'impressions' => $metrics['views'] ?? 0,
                    'engagement' => ($metrics['likes'] ?? 0) + ($metrics['comments'] ?? 0) + ($metrics['shares'] ?? 0)
                ];
            }
            
            return [
                'total_followers' => $totalFollowers,
                'total_posts' => $totalPosts,
                'engagement_rate' => $engagementRate,
                'total_likes' => $totalLikes,
                'total_comments' => $totalComments,
                'total_shares' => $totalShares,
                'total_saves' => $totalSaves,
                'total_impressions' => $totalImpressions,
                'follower_growth' => $followerGrowth,
                'top_content_type' => $topContentType,
                'best_posting_time' => $bestPostingTime,
                'top_hashtags' => $topHashtags,
                'personal_best_posts' => $personalBestPostsData,
                'time_series' => $timeSeriesData,
                'labels' => $timeSeriesData['labels'],
                'data' => $timeSeriesData['impressions']
            ];
        });
    }
    
    /**
     * Get time series data for charts
     *
     * @param \Illuminate\Database\Eloquent\Builder $postsQuery
     * @param int $days
     * @return array
     */
    protected function getTimeSeriesData($postsQuery, $days)
    {
        // Get views data for the last N days
        $viewsData = $postsQuery->selectRaw('DATE(published_at) as date,
            SUM(JSON_EXTRACT(metrics, "$.views")) as impressions,
            SUM(JSON_EXTRACT(metrics, "$.likes")) as likes,
            SUM(JSON_EXTRACT(metrics, "$.comments")) as comments,
            SUM(JSON_EXTRACT(metrics, "$.shares")) as shares')
            ->groupBy('date')
            ->orderBy('date')
            ->get();
        
        // Format data for Chart.js
        $labels = [];
        $impressions = [];
        $engagements = [];
        
        // Fill in missing dates
        $startDate = now()->subDays($days);
        for ($i = 0; $i <= $days; $i++) {
            $date = $startDate->copy()->addDays($i)->format('Y-m-d');
            $labels[] = $date;
            $impressions[] = 0;
            $engagements[] = 0;
        }
        
        // Fill in actual data
        foreach ($viewsData as $item) {
            $index = array_search($item->date, $labels);
            if ($index !== false) {
                $impressions[$index] = (int) $item->impressions;
                $engagements[$index] = (int) ($item->likes + $item->comments + $item->shares);
            }
        }
        
        return [
            'labels' => $labels,
            'impressions' => $impressions,
            'engagements' => $engagements
        ];
    }
    
    /**
     * Get top content type based on engagement
     *
     * @param \Illuminate\Database\Eloquent\Collection $posts
     * @return string
     */
    protected function getTopContentType($posts)
    {
        $contentTypes = [
            'post' => 0,
            'reel' => 0,
            'story' => 0,
            'video' => 0
        ];
        
        foreach ($posts as $post) {
            // Determine content type based on post data
            $type = 'post'; // default
            
            // Check if it's a video
            if (!empty($post->media['video'])) {
                $type = 'video';
            }
            // Check if it's a story (based on provider and other heuristics)
            elseif (isset($post->raw_json['is_story']) && $post->raw_json['is_story']) {
                $type = 'story';
            }
            // Check if it's a reel (Instagram-specific)
            elseif (isset($post->raw_json['is_reel']) && $post->raw_json['is_reel']) {
                $type = 'reel';
            }
            
            // Calculate engagement for this post
            $metrics = $post->metrics ?? [];
            $engagement = ($metrics['likes'] ?? 0) + ($metrics['comments'] ?? 0) + ($metrics['shares'] ?? 0) + ($metrics['saves'] ?? 0);
            
            // Add engagement to content type
            $contentTypes[$type] += $engagement;
        }
        
        // Return the content type with highest engagement
        return array_keys($contentTypes, max($contentTypes))[0];
    }
    
    /**
     * Get best posting time based on engagement
     *
     * @param \Illuminate\Database\Eloquent\Collection $posts
     * @return string
     */
    protected function getBestPostingTime($posts)
    {
        $timeSlots = [];
        
        foreach ($posts as $post) {
            if ($post->published_at) {
                $hour = $post->published_at->format('H');
                
                // Calculate engagement for this post
                $metrics = $post->metrics ?? [];
                $engagement = ($metrics['likes'] ?? 0) + ($metrics['comments'] ?? 0) + ($metrics['shares'] ?? 0) + ($metrics['saves'] ?? 0);
                
                // Add engagement to time slot
                $timeSlots[$hour] = ($timeSlots[$hour] ?? 0) + $engagement;
            }
        }
        
        if (empty($timeSlots)) {
            return 'Not enough data';
        }
        
        // Find the hour with the highest engagement
        $bestHour = array_keys($timeSlots, max($timeSlots))[0];
        return $bestHour . ':00 - ' . ($bestHour + 1) . ':00';
    }
    
    /**
     * Get top hashtags from posts
     *
     * @param \Illuminate\Database\Eloquent\Collection $posts
     * @return array
     */
    protected function getTopHashtags($posts)
    {
        $hashtags = [];
        
        foreach ($posts as $post) {
            // Combine caption and content to search for hashtags
            $text = ($post->caption ?? '') . ' ' . ($post->content ?? '');
            
            // Extract hashtags using regex
            preg_match_all('/#(\w+)/', $text, $matches);
            
            foreach ($matches[0] as $hashtag) {
                $hashtag = strtolower($hashtag);
                $hashtags[$hashtag] = ($hashtags[$hashtag] ?? 0) + 1;
            }
        }
        
        // Sort by frequency and return top 10
        arsort($hashtags);
        return array_slice(array_keys($hashtags), 0, 10);
    }
}