<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\KnowledgeBaseDocument;
use App\Models\Brand;
use App\Services\RAGService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Exception;

class KnowledgeBaseController extends Controller
{
    protected RAGService $ragService;

    public function __construct(RAGService $ragService)
    {
        $this->ragService = $ragService;
    }

    /**
     * Display knowledge base management interface.
     */
    public function index(Request $request, Brand $brand)
    {
        Gate::authorize('manage-support', $brand);

        $documents = KnowledgeBaseDocument::where('brand_id', $brand->id)
            ->with('uploader')
            ->when($request->input('type'), function ($query, $type) {
                $query->ofType($type);
            })
            ->when($request->input('status'), function ($query, $status) {
                $query->where('status', $status);
            })
            ->when($request->input('search'), function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('content', 'like', "%{$search}%");
                });
            })
            ->latest()
            ->paginate(20);

        $stats = [
            'total' => KnowledgeBaseDocument::where('brand_id', $brand->id)->count(),
            'active' => KnowledgeBaseDocument::where('brand_id', $brand->id)->active()->count(),
            'needs_indexing' => KnowledgeBaseDocument::where('brand_id', $brand->id)
                ->active()
                ->get()
                ->filter(fn($doc) => $doc->needsReindexing())
                ->count(),
        ];

        return view('brand.knowledge-base.index', compact('brand', 'documents', 'stats'));
    }

    /**
     * Store new knowledge base document.
     */
    public function store(Request $request, Brand $brand): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'document_type' => ['required', 'string', Rule::in(['faq', 'manual', 'policy', 'terms', 'campaign', 'other'])],
            'content' => 'required|string',
            'metadata' => 'nullable|array',
            'file' => 'nullable|file|mimes:pdf,docx,txt|max:10240',
        ]);

        try {
            $filePath = null;
            $fileType = null;
            $fileSize = null;

            if ($request->hasFile('file')) {
                $file = $request->file('file');
                $filePath = $file->store('knowledge-base/' . $brand->id, 'private');
                $fileType = $file->getClientOriginalExtension();
                $fileSize = $file->getSize();
            }

            $document = KnowledgeBaseDocument::create([
                'brand_id' => $brand->id,
                'title' => $validated['title'],
                'document_type' => $validated['document_type'],
                'content' => $validated['content'],
                'metadata' => $validated['metadata'] ?? null,
                'file_path' => $filePath,
                'file_type' => $fileType,
                'file_size' => $fileSize,
                'status' => 'active',
                'uploaded_by' => $request->user()->id,
            ]);

            // Index the document for RAG
            $this->ragService->indexDocument($document);

            return response()->json([
                'success' => true,
                'message' => 'Knowledge base document created and indexed successfully',
                'data' => $document->load('uploader'),
            ], 201);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create knowledge base document',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Show specific document.
     */
    public function show(Brand $brand, KnowledgeBaseDocument $document): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        if ($document->brand_id !== $brand->id) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $document->load('uploader'),
        ]);
    }

    /**
     * Update document.
     */
    public function update(Request $request, Brand $brand, KnowledgeBaseDocument $document): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        if ($document->brand_id !== $brand->id) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found',
            ], 404);
        }

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'document_type' => ['sometimes', 'required', 'string', Rule::in(['faq', 'manual', 'policy', 'terms', 'campaign', 'other'])],
            'content' => 'sometimes|required|string',
            'metadata' => 'nullable|array',
            'status' => ['sometimes', 'required', 'string', Rule::in(['active', 'archived', 'draft'])],
        ]);

        try {
            $document->update($validated);

            // Re-index if content changed
            if (isset($validated['content']) || isset($validated['title'])) {
                $this->ragService->indexDocument($document);
            }

            return response()->json([
                'success' => true,
                'message' => 'Document updated successfully',
                'data' => $document->fresh()->load('uploader'),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update document',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete document.
     */
    public function destroy(Brand $brand, KnowledgeBaseDocument $document): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        if ($document->brand_id !== $brand->id) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found',
            ], 404);
        }

        try {
            // Delete file if exists
            if ($document->file_path) {
                Storage::disk('private')->delete($document->file_path);
            }

            $document->delete();

            return response()->json([
                'success' => true,
                'message' => 'Document deleted successfully',
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete document',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Batch index documents.
     */
    public function batchIndex(Brand $brand): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        try {
            $results = $this->ragService->batchIndexDocuments($brand->id);

            return response()->json([
                'success' => true,
                'message' => 'Batch indexing completed',
                'data' => $results,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Batch indexing failed',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Re-index specific document.
     */
    public function reindex(Brand $brand, KnowledgeBaseDocument $document): JsonResponse
    {
        Gate::authorize('manage-support', $brand);

        if ($document->brand_id !== $brand->id) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found',
            ], 404);
        }

        try {
            $this->ragService->indexDocument($document);

            return response()->json([
                'success' => true,
                'message' => 'Document re-indexed successfully',
                'data' => $document->fresh(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Re-indexing failed',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Download document file.
     */
    public function download(Brand $brand, KnowledgeBaseDocument $document)
    {
        Gate::authorize('manage-support', $brand);

        if ($document->brand_id !== $brand->id || !$document->file_path) {
            abort(404);
        }

        return Storage::disk('private')->download(
            $document->file_path,
            $document->title . '.' . $document->file_type
        );
    }
}