<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\CampaignFinancial;
use App\Models\FinancialReport;
use App\Models\BudgetScenario;
use App\Services\FinancialInsightsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;

class FinancialInsightsController extends Controller
{
    private FinancialInsightsService $financialService;

    public function __construct(FinancialInsightsService $financialService)
    {
        $this->middleware('auth');
        $this->middleware('role:brand');
        $this->financialService = $financialService;
    }

    /**
     * Display financial dashboard
     */
    public function index(Request $request)
    {
        $brand = auth()->user();
        
        // Get date range from request or default to last 30 days
        $periodStart = $request->input('start_date') 
            ? Carbon::parse($request->input('start_date'))
            : now()->subDays(30);
        
        $periodEnd = $request->input('end_date')
            ? Carbon::parse($request->input('end_date'))
            : now();

        // Get dashboard data
        $dashboardData = $this->financialService->getDashboardData(
            $brand,
            $periodStart,
            $periodEnd
        );

        return view('brand.financial-insights.index', [
            'data' => $dashboardData,
            'periodStart' => $periodStart,
            'periodEnd' => $periodEnd,
        ]);
    }

    /**
     * Update campaign financial data
     */
    public function updateCampaignFinancials(Request $request, Campaign $campaign)
    {
        Gate::authorize('update', $campaign);

        $validated = $request->validate([
            'spend_data' => 'required|array',
            'spend_data.ad_spend' => 'required|numeric|min:0',
            'spend_data.influencer_payments' => 'required|numeric|min:0',
            'spend_data.production_costs' => 'required|numeric|min:0',
            'spend_data.other_costs' => 'nullable|numeric|min:0',
            'spend_data.breakdown' => 'nullable|array',
            'performance_data' => 'required|array',
            'performance_data.clicks' => 'required|integer|min:0',
            'performance_data.impressions' => 'required|integer|min:0',
            'performance_data.conversions' => 'required|integer|min:0',
            'performance_data.avg_order_value' => 'required|numeric|min:0',
            'performance_data.metrics' => 'nullable|array',
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);

        $financial = $this->financialService->updateCampaignFinancials(
            $campaign,
            $validated['spend_data'],
            $validated['performance_data'],
            Carbon::parse($validated['period_start']),
            Carbon::parse($validated['period_end'])
        );

        return response()->json([
            'success' => true,
            'message' => 'Campaign financial data updated successfully',
            'data' => $financial,
        ]);
    }

    /**
     * Generate campaign financial report
     */
    public function generateCampaignReport(Request $request, Campaign $campaign)
    {
        Gate::authorize('view', $campaign);

        $validated = $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
            'run_ai_analysis' => 'boolean',
        ]);

        $report = $this->financialService->generateCampaignReport(
            $campaign,
            Carbon::parse($validated['period_start']),
            Carbon::parse($validated['period_end']),
            $validated['run_ai_analysis'] ?? true
        );

        return response()->json([
            'success' => true,
            'message' => 'Campaign report generated successfully',
            'data' => $report->load('campaign'),
        ]);
    }

    /**
     * Generate brand consolidated report
     */
    public function generateBrandReport(Request $request)
    {
        $validated = $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
            'run_ai_analysis' => 'boolean',
        ]);

        $report = $this->financialService->generateBrandReport(
            auth()->user(),
            Carbon::parse($validated['period_start']),
            Carbon::parse($validated['period_end']),
            $validated['run_ai_analysis'] ?? true
        );

        return response()->json([
            'success' => true,
            'message' => 'Brand report generated successfully',
            'data' => $report,
        ]);
    }

    /**
     * View specific financial report
     */
    public function showReport(FinancialReport $report)
    {
        Gate::authorize('view', $report);

        return view('brand.financial-insights.report', [
            'report' => $report->load(['campaign', 'budgetScenarios']),
        ]);
    }

    /**
     * Get financial insights summary
     */
    public function insights()
    {
        $insights = $this->financialService->getInsightsSummary(auth()->user());

        return response()->json([
            'success' => true,
            'data' => $insights,
        ]);
    }

    /**
     * Generate budget scenarios
     */
    public function generateScenarios(Request $request, FinancialReport $report)
    {
        Gate::authorize('view', $report);

        $validated = $request->validate([
            'total_budget' => 'required|numeric|min:0',
        ]);

        $scenarios = $this->financialService->generateScenarios(
            $report,
            $validated['total_budget']
        );

        return response()->json([
            'success' => true,
            'message' => 'Budget scenarios generated successfully',
            'data' => $scenarios,
        ]);
    }

    /**
     * View budget scenarios
     */
    public function scenarios(Request $request)
    {
        $scenarios = BudgetScenario::where('brand_id', auth()->id())
            ->where('status', '!=', 'archived')
            ->with('financialReport')
            ->latest()
            ->paginate(10);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'data' => $scenarios,
            ]);
        }

        return view('brand.financial-insights.scenarios', [
            'scenarios' => $scenarios,
        ]);
    }

    /**
     * View specific budget scenario
     */
    public function showScenario(BudgetScenario $scenario)
    {
        Gate::authorize('view', $scenario);

        return view('brand.financial-insights.scenario-detail', [
            'scenario' => $scenario->load('financialReport'),
        ]);
    }

    /**
     * Apply budget scenario
     */
    public function applyScenario(BudgetScenario $scenario)
    {
        Gate::authorize('update', $scenario);

        $success = $this->financialService->applyScenario($scenario);

        return response()->json([
            'success' => $success,
            'message' => $success 
                ? 'Budget scenario applied successfully' 
                : 'Failed to apply budget scenario',
            'data' => $scenario->fresh(),
        ]);
    }

    /**
     * Archive budget scenario
     */
    public function archiveScenario(BudgetScenario $scenario)
    {
        Gate::authorize('delete', $scenario);

        $scenario->archive();

        return response()->json([
            'success' => true,
            'message' => 'Budget scenario archived successfully',
        ]);
    }

    /**
     * Compare two financial periods
     */
    public function comparePeriods(Request $request)
    {
        $validated = $request->validate([
            'period1_start' => 'required|date',
            'period1_end' => 'required|date|after:period1_start',
            'period2_start' => 'required|date',
            'period2_end' => 'required|date|after:period2_start',
        ]);

        $comparison = $this->financialService->comparePeriods(
            auth()->user(),
            Carbon::parse($validated['period1_start']),
            Carbon::parse($validated['period1_end']),
            Carbon::parse($validated['period2_start']),
            Carbon::parse($validated['period2_end'])
        );

        return response()->json([
            'success' => true,
            'data' => $comparison,
        ]);
    }

    /**
     * Export financial report as PDF
     */
    public function exportPDF(FinancialReport $report)
    {
        Gate::authorize('view', $report);

        $exportData = $this->financialService->exportReport($report, 'pdf');

        $pdf = Pdf::loadView('brand.financial-insights.pdf', [
            'report' => $report->load(['campaign', 'budgetScenarios']),
        ]);

        // Add watermark
        $pdf->setOption('footer-html', view('brand.financial-insights.pdf-footer', [
            'disclaimer' => 'AI-generated estimates. Not financial advice.',
        ])->render());

        return $pdf->download($exportData['filename']);
    }

    /**
     * Export financial report as Excel
     */
    public function exportExcel(FinancialReport $report)
    {
        Gate::authorize('view', $report);

        $exportData = $this->financialService->exportReport($report, 'excel');

        // Create Excel export (simplified - you'd need to create an Export class)
        return response()->json([
            'success' => true,
            'message' => 'Excel export prepared',
            'download_url' => route('brand.financial-insights.download-excel', $report),
        ]);
    }

    /**
     * Get ROI trends chart data
     */
    public function roiTrends(Request $request)
    {
        $validated = $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
            'granularity' => 'in:daily,weekly,monthly',
        ]);

        $granularity = $validated['granularity'] ?? 'weekly';
        $periodStart = Carbon::parse($validated['period_start']);
        $periodEnd = Carbon::parse($validated['period_end']);

        $financials = CampaignFinancial::where('brand_id', auth()->id())
            ->inPeriod($periodStart, $periodEnd)
            ->orderBy('period_start')
            ->get();

        // Group by granularity and calculate ROI
        $chartData = $this->groupFinancialsByGranularity($financials, $granularity);

        return response()->json([
            'success' => true,
            'data' => $chartData,
        ]);
    }

    /**
     * Get spend vs revenue chart data
     */
    public function spendVsRevenue(Request $request)
    {
        $validated = $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);

        $periodStart = Carbon::parse($validated['period_start']);
        $periodEnd = Carbon::parse($validated['period_end']);

        $financials = CampaignFinancial::where('brand_id', auth()->id())
            ->inPeriod($periodStart, $periodEnd)
            ->with('campaign')
            ->get();

        $chartData = $financials->map(function ($financial) {
            return [
                'campaign' => $financial->campaign->name ?? 'Unknown',
                'spend' => $financial->total_spend,
                'revenue' => $financial->total_revenue,
                'profit' => $financial->profit,
                'roi' => $financial->roi_percentage,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $chartData,
        ]);
    }

    /**
     * Get channel performance data
     */
    public function channelPerformance(Request $request)
    {
        $validated = $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);

        $periodStart = Carbon::parse($validated['period_start']);
        $periodEnd = Carbon::parse($validated['period_end']);

        $report = FinancialReport::where('brand_id', auth()->id())
            ->where('report_type', 'brand_consolidated')
            ->where('period_start', '>=', $periodStart)
            ->where('period_end', '<=', $periodEnd)
            ->latest()
            ->first();

        if (!$report || !$report->spend_breakdown) {
            return response()->json([
                'success' => false,
                'message' => 'No channel performance data available',
            ]);
        }

        $channelData = [];
        foreach ($report->spend_breakdown as $channel => $spend) {
            $revenue = $report->revenue_breakdown[$channel] ?? 0;
            $roi = $spend > 0 ? (($revenue - $spend) / $spend) * 100 : 0;

            $channelData[] = [
                'channel' => ucfirst($channel),
                'spend' => $spend,
                'revenue' => $revenue,
                'roi' => round($roi, 2),
            ];
        }

        return response()->json([
            'success' => true,
            'data' => $channelData,
        ]);
    }

    /**
     * Group financials by granularity for charts
     */
    private function groupFinancialsByGranularity(
        $financials,
        string $granularity
    ): array {
        $grouped = [];

        foreach ($financials as $financial) {
            $key = match($granularity) {
                'daily' => $financial->period_start->format('Y-m-d'),
                'weekly' => $financial->period_start->startOfWeek()->format('Y-m-d'),
                'monthly' => $financial->period_start->format('Y-m'),
                default => $financial->period_start->format('Y-m-d'),
            };

            if (!isset($grouped[$key])) {
                $grouped[$key] = [
                    'period' => $key,
                    'total_spend' => 0,
                    'total_revenue' => 0,
                    'conversions' => 0,
                ];
            }

            $grouped[$key]['total_spend'] += $financial->total_spend;
            $grouped[$key]['total_revenue'] += $financial->total_revenue;
            $grouped[$key]['conversions'] += $financial->conversions;
        }

        // Calculate ROI for each period
        return array_values(array_map(function ($data) {
            $data['roi'] = $data['total_spend'] > 0 
                ? (($data['total_revenue'] - $data['total_spend']) / $data['total_spend']) * 100 
                : 0;
            return $data;
        }, $grouped));
    }
}