<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Jobs\GenerateProposalJob;
use App\Models\AiJob;
use App\Models\ProposalRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AiJobController extends Controller
{
    /**
     * Display a listing of AI jobs.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->authorize('admin');

        $aiJobs = AiJob::with(['brand', 'creator'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.ai-jobs.index', compact('aiJobs'));
    }

    /**
     * Display the specified AI job.
     *
     * @param  string  $jobId
     * @return \Illuminate\Http\Response
     */
    public function show($jobId)
    {
        $this->authorize('admin');

        $aiJob = AiJob::where('job_id', $jobId)
            ->with(['brand', 'creator', 'proposal'])
            ->firstOrFail();

        return view('admin.ai-jobs.show', compact('aiJob'));
    }

    /**
     * Re-run the AI job.
     *
     * @param  string  $jobId
     * @return \Illuminate\Http\Response
     */
    public function rerun($jobId)
    {
        $this->authorize('admin');

        $aiJob = AiJob::where('job_id', $jobId)->firstOrFail();

        // Reset job status
        $aiJob->update([
            'status' => 'queued',
            'result_text' => null,
            'result_meta' => null,
            'provider' => null,
            'error_message' => null,
            'token_usage' => 0,
            'completed_at' => null,
        ]);

        // Get the original payload from the proposal request
        $proposalRequest = ProposalRequest::where('ai_job_id', $aiJob->id)->first();
        if (!$proposalRequest) {
            return redirect()->back()->with('error', 'Proposal request not found.');
        }

        $payload = [
            'campaign_brief' => $proposalRequest->campaign_brief,
            'tone' => $proposalRequest->tone,
            'budget_range' => $proposalRequest->budget_range,
            'deliverables' => $proposalRequest->deliverables,
            'brand_name' => $aiJob->brand->name,
            'brand_industry' => $aiJob->brand->settings['industry'] ?? 'general',
        ];

        // Dispatch the actual job
        GenerateProposalJob::dispatch($aiJob->id, $payload);

        return redirect()->back()->with('success', 'AI job re-run successfully!');
    }

    /**
     * Get the logs for an AI job.
     *
     * @param  string  $jobId
     * @return \Illuminate\Http\Response
     */
    public function logs($jobId)
    {
        $this->authorize('admin');

        // In a real implementation, you would retrieve logs from your logging system
        // For now, we'll return a placeholder response
        $logs = [
            'Job ID: ' . $jobId,
            'Status: ' . AiJob::where('job_id', $jobId)->first()->status,
            'Timestamp: ' . now()->format('Y-m-d H:i:s'),
            'Message: Job processed successfully',
        ];

        return response()->json([
            'logs' => $logs,
        ]);
    }
}