<?php

namespace App\Console\Commands;

use App\Jobs\SyncUserSocialAccounts;
use App\Jobs\SyncBrandSocialAccounts;
use App\Models\User;
use App\Models\Brand;
use Illuminate\Console\Command;

class SocialSyncCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'social:sync {--user=} {--brand=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync social accounts for a user, brand, or all users';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Get the user ID from the option
        $userId = $this->option('user');
        
        // Get the brand ID from the option
        $brandId = $this->option('brand');

        if ($userId) {
            // Sync specific user
            $user = User::find($userId);
            
            if (!$user) {
                $this->error('User not found!');
                return 1;
            }
            
            $this->info('Syncing social accounts for user: ' . $user->name);
            SyncUserSocialAccounts::dispatch($user);
        } elseif ($brandId) {
            // Sync specific brand
            $brand = Brand::find($brandId);
            
            if (!$brand) {
                $this->error('Brand not found!');
                return 1;
            }
            
            $this->info('Syncing social accounts for brand: ' . $brand->name);
            SyncBrandSocialAccounts::dispatch($brand);
        } else {
            // Sync all users with social accounts
            $users = User::whereHas('socialAccounts')->get();
            
            if ($users->isNotEmpty()) {
                $this->info('Syncing social accounts for ' . $users->count() . ' users...');
                
                foreach ($users as $user) {
                    $this->line('Syncing for user: ' . $user->name);
                    SyncUserSocialAccounts::dispatch($user);
                }
            }
            
            // Sync all brands with social accounts
            $brands = Brand::whereHas('socialAccounts')->get();
            
            if ($brands->isNotEmpty()) {
                $this->info('Syncing social accounts for ' . $brands->count() . ' brands...');
                
                foreach ($brands as $brand) {
                    $this->line('Syncing for brand: ' . $brand->name);
                    SyncBrandSocialAccounts::dispatch($brand);
                }
            }
            
            if ($users->isEmpty() && $brands->isEmpty()) {
                $this->info('No users or brands with social accounts found.');
                return 0;
            }
        }

        $this->info('Sync jobs dispatched successfully!');
        return 0;
    }
}
