<?php

namespace App\Console\Commands;

use App\Models\ProviderCredential;
use App\Models\User;
use App\Notifications\ProviderTokenExpiringNotification;
use App\Services\ProviderCredentialService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckProviderTokenExpiry extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'providers:check-expiry 
                            {--days=7 : Number of days before expiry to send alert}
                            {--auto-refresh : Automatically refresh tokens if possible}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for expiring provider tokens and send notifications';

    protected $credentialService;

    /**
     * Create a new command instance.
     */
    public function __construct(ProviderCredentialService $credentialService)
    {
        parent::__construct();
        $this->credentialService = $credentialService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = (int) $this->option('days');
        $autoRefresh = $this->option('auto-refresh');

        $this->info("Checking for tokens expiring within {$days} days...");

        // Get expiring credentials
        $expiringCredentials = ProviderCredential::expiring($days)
            ->where('expiry_alert_sent', false)
            ->with(['brand', 'creator'])
            ->get();

        $this->info("Found {$expiringCredentials->count()} expiring credentials");

        $refreshed = 0;
        $notified = 0;
        $failed = 0;

        foreach ($expiringCredentials as $credential) {
            $this->line("Processing: {$credential->provider} for brand {$credential->brand->name}");

            // Try to auto-refresh if enabled and refresh token exists
            if ($autoRefresh && $credential->refresh_token) {
                $this->line("  Attempting to refresh token...");
                
                if ($this->credentialService->refreshToken($credential)) {
                    $this->info("  ✓ Token refreshed successfully");
                    $refreshed++;
                    continue;
                } else {
                    $this->warn("  ✗ Token refresh failed");
                }
            }

            // Send notification to brand admins
            $brandAdmins = User::where('brand_id', $credential->brand_id)
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'brand_admin');
                })
                ->get();

            if ($brandAdmins->isEmpty()) {
                $this->warn("  No brand admins found for brand {$credential->brand->name}");
                $failed++;
                continue;
            }

            foreach ($brandAdmins as $admin) {
                try {
                    $admin->notify(new ProviderTokenExpiringNotification($credential, $days));
                    $this->info("  ✓ Notification sent to {$admin->email}");
                } catch (\Exception $e) {
                    $this->error("  ✗ Failed to send notification to {$admin->email}: {$e->getMessage()}");
                    Log::error("Failed to send token expiry notification", [
                        'credential_id' => $credential->id,
                        'admin_email' => $admin->email,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            // Mark alert as sent
            $credential->update([
                'expiry_alert_sent' => true,
                'status' => 'expiring',
            ]);

            $notified++;
        }

        // Check for already expired credentials
        $expiredCredentials = ProviderCredential::expired()
            ->where('status', '!=', 'failed')
            ->get();

        if ($expiredCredentials->isNotEmpty()) {
            $this->warn("\nFound {$expiredCredentials->count()} expired credentials");
            
            foreach ($expiredCredentials as $credential) {
                $credential->update(['status' => 'failed']);
                $this->warn("  Marked as failed: {$credential->provider} for brand {$credential->brand->name}");
            }
        }

        // Summary
        $this->newLine();
        $this->info("Summary:");
        $this->table(
            ['Action', 'Count'],
            [
                ['Tokens Refreshed', $refreshed],
                ['Notifications Sent', $notified],
                ['Failed', $failed],
                ['Expired Credentials', $expiredCredentials->count()],
            ]
        );

        return Command::SUCCESS;
    }
}