"""
Pytest configuration and fixtures
"""
import pytest
import asyncio
from typing import AsyncGenerator
from sqlalchemy.ext.asyncio import create_async_engine, AsyncSession, async_sessionmaker
from httpx import AsyncClient

from app.main import app
from app.database import Base, get_db
from app.models import AIJob, PublishJob
from config import settings


# Set test mode
settings.test_mode = True
settings.mock_providers = True


@pytest.fixture(scope="session")
def event_loop():
    """Create event loop for async tests"""
    loop = asyncio.get_event_loop_policy().new_event_loop()
    yield loop
    loop.close()


@pytest.fixture(scope="function")
async def test_db():
    """Create test database"""
    # Use in-memory SQLite for tests
    engine = create_async_engine(
        "sqlite+aiosqlite:///:memory:",
        echo=False
    )
    
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.create_all)
    
    AsyncTestSession = async_sessionmaker(
        engine,
        class_=AsyncSession,
        expire_on_commit=False
    )
    
    async def override_get_db():
        async with AsyncTestSession() as session:
            yield session
    
    app.dependency_overrides[get_db] = override_get_db
    
    yield AsyncTestSession
    
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.drop_all)
    
    await engine.dispose()


@pytest.fixture
async def client(test_db) -> AsyncGenerator[AsyncClient, None]:
    """Create test client"""
    async with AsyncClient(app=app, base_url="http://test") as ac:
        yield ac


@pytest.fixture
def api_headers():
    """API headers with auth"""
    return {
        "X-API-Key": settings.laravel_api_key,
        "Content-Type": "application/json"
    }


@pytest.fixture
def admin_headers():
    """Admin headers with auth"""
    return {
        "X-Admin-Key": settings.secret_key,
        "Content-Type": "application/json"
    }


@pytest.fixture
async def sample_ai_job(test_db):
    """Create sample AI job"""
    async with test_db() as db:
        job = AIJob(
            job_type="content_ideas",
            tenant_id="test-tenant",
            owner_id="test-user",
            payload={"topic": "AI in marketing", "count": 5},
            status="pending"
        )
        db.add(job)
        await db.commit()
        await db.refresh(job)
        return job


@pytest.fixture
async def sample_publish_job(test_db):
    """Create sample publish job"""
    async with test_db() as db:
        job = PublishJob(
            tenant_id="test-tenant",
            owner_id="test-user",
            platform="twitter",
            content={"text": "Test tweet"},
            status="pending"
        )
        db.add(job)
        await db.commit()
        await db.refresh(job)
        return job