"""
AI Provider adapters for OpenAI, Anthropic, and local LLM
"""
from typing import Dict, Any, Optional, List
from abc import ABC, abstractmethod
import time
import tiktoken

from config import settings
from app.utils.observability import logger, cost_tracker, metrics_collector
from app.utils.security import pii_redactor


class AIProvider(ABC):
    """Base class for AI providers"""
    
    def __init__(self, provider_name: str):
        self.provider_name = provider_name
    
    @abstractmethod
    async def generate_completion(
        self,
        prompt: str,
        max_tokens: int = 1000,
        temperature: float = 0.7,
        **kwargs
    ) -> Dict[str, Any]:
        """Generate text completion"""
        pass
    
    @abstractmethod
    async def generate_embedding(self, text: str) -> List[float]:
        """Generate text embedding"""
        pass
    
    def count_tokens(self, text: str, model: str) -> int:
        """Count tokens in text"""
        try:
            encoding = tiktoken.encoding_for_model(model)
            return len(encoding.encode(text))
        except Exception:
            # Fallback: rough estimate
            return len(text.split()) * 1.3


class OpenAIProvider(AIProvider):
    """OpenAI API provider"""
    
    def __init__(self):
        super().__init__("openai")
        self.api_key = settings.openai_api_key
        self.default_model = settings.openai_default_model
        self.embedding_model = settings.openai_embedding_model
        
        if not self.api_key:
            raise ValueError("OpenAI API key not configured")
    
    async def generate_completion(
        self,
        prompt: str,
        max_tokens: int = 1000,
        temperature: float = 0.7,
        model: Optional[str] = None,
        **kwargs
    ) -> Dict[str, Any]:
        """Generate completion using OpenAI"""
        from openai import AsyncOpenAI
        
        client = AsyncOpenAI(api_key=self.api_key)
        model = model or self.default_model
        
        # Redact PII from prompt
        safe_prompt = pii_redactor.redact_text(prompt)
        
        start_time = time.time()
        
        try:
            response = await client.chat.completions.create(
                model=model,
                messages=[{"role": "user", "content": safe_prompt}],
                max_tokens=max_tokens,
                temperature=temperature,
                **kwargs
            )
            
            latency = time.time() - start_time
            
            # Extract response
            content = response.choices[0].message.content
            tokens_input = response.usage.prompt_tokens
            tokens_output = response.usage.completion_tokens
            
            # Calculate cost
            cost = cost_tracker.calculate_cost(
                self.provider_name,
                model,
                tokens_input,
                tokens_output
            )
            
            # Track metrics
            metrics_collector.track_llm_request(
                self.provider_name,
                model,
                tokens_input,
                tokens_output,
                cost,
                latency
            )
            
            logger.info(
                "OpenAI completion generated",
                model=model,
                tokens_input=tokens_input,
                tokens_output=tokens_output,
                cost_usd=cost,
                latency=latency
            )
            
            return {
                "content": content,
                "tokens_input": tokens_input,
                "tokens_output": tokens_output,
                "tokens_total": tokens_input + tokens_output,
                "cost_usd": cost,
                "model": model,
                "provider": self.provider_name
            }
            
        except Exception as e:
            logger.error("OpenAI completion failed", error=str(e), model=model)
            raise
    
    async def generate_embedding(self, text: str, model: Optional[str] = None) -> List[float]:
        """Generate embedding using OpenAI"""
        from openai import AsyncOpenAI
        
        client = AsyncOpenAI(api_key=self.api_key)
        model = model or self.embedding_model
        
        try:
            response = await client.embeddings.create(
                model=model,
                input=text
            )
            
            embedding = response.data[0].embedding
            tokens_used = response.usage.total_tokens
            
            # Calculate cost
            cost = cost_tracker.calculate_cost(
                self.provider_name,
                model,
                tokens_used,
                0
            )
            
            logger.info(
                "OpenAI embedding generated",
                model=model,
                tokens=tokens_used,
                cost_usd=cost
            )
            
            return embedding
            
        except Exception as e:
            logger.error("OpenAI embedding failed", error=str(e), model=model)
            raise


class AnthropicProvider(AIProvider):
    """Anthropic Claude API provider"""
    
    def __init__(self):
        super().__init__("anthropic")
        self.api_key = settings.anthropic_api_key
        self.default_model = settings.anthropic_default_model
        
        if not self.api_key:
            raise ValueError("Anthropic API key not configured")
    
    async def generate_completion(
        self,
        prompt: str,
        max_tokens: int = 1000,
        temperature: float = 0.7,
        model: Optional[str] = None,
        **kwargs
    ) -> Dict[str, Any]:
        """Generate completion using Anthropic Claude"""
        from anthropic import AsyncAnthropic
        
        client = AsyncAnthropic(api_key=self.api_key)
        model = model or self.default_model
        
        # Redact PII from prompt
        safe_prompt = pii_redactor.redact_text(prompt)
        
        start_time = time.time()
        
        try:
            response = await client.messages.create(
                model=model,
                max_tokens=max_tokens,
                temperature=temperature,
                messages=[{"role": "user", "content": safe_prompt}],
                **kwargs
            )
            
            latency = time.time() - start_time
            
            # Extract response
            content = response.content[0].text
            tokens_input = response.usage.input_tokens
            tokens_output = response.usage.output_tokens
            
            # Calculate cost
            cost = cost_tracker.calculate_cost(
                self.provider_name,
                model,
                tokens_input,
                tokens_output
            )
            
            # Track metrics
            metrics_collector.track_llm_request(
                self.provider_name,
                model,
                tokens_input,
                tokens_output,
                cost,
                latency
            )
            
            logger.info(
                "Anthropic completion generated",
                model=model,
                tokens_input=tokens_input,
                tokens_output=tokens_output,
                cost_usd=cost,
                latency=latency
            )
            
            return {
                "content": content,
                "tokens_input": tokens_input,
                "tokens_output": tokens_output,
                "tokens_total": tokens_input + tokens_output,
                "cost_usd": cost,
                "model": model,
                "provider": self.provider_name
            }
            
        except Exception as e:
            logger.error("Anthropic completion failed", error=str(e), model=model)
            raise
    
    async def generate_embedding(self, text: str, model: Optional[str] = None) -> List[float]:
        """Anthropic doesn't provide embeddings, use OpenAI as fallback"""
        logger.warning("Anthropic doesn't support embeddings, using OpenAI fallback")
        openai_provider = OpenAIProvider()
        return await openai_provider.generate_embedding(text)


class LocalLLMProvider(AIProvider):
    """Local LLM provider (e.g., Ollama)"""
    
    def __init__(self):
        super().__init__("local")
        self.base_url = settings.local_llm_url
        self.default_model = settings.local_llm_model
        
        if not settings.local_llm_enabled:
            raise ValueError("Local LLM not enabled")
    
    async def generate_completion(
        self,
        prompt: str,
        max_tokens: int = 1000,
        temperature: float = 0.7,
        model: Optional[str] = None,
        **kwargs
    ) -> Dict[str, Any]:
        """Generate completion using local LLM"""
        import httpx
        
        model = model or self.default_model
        start_time = time.time()
        
        try:
            async with httpx.AsyncClient(timeout=120.0) as client:
                response = await client.post(
                    f"{self.base_url}/api/generate",
                    json={
                        "model": model,
                        "prompt": prompt,
                        "options": {
                            "temperature": temperature,
                            "num_predict": max_tokens
                        }
                    }
                )
                response.raise_for_status()
                data = response.json()
            
            latency = time.time() - start_time
            content = data.get("response", "")
            
            # Estimate tokens (local LLM may not provide exact counts)
            tokens_input = self.count_tokens(prompt, model)
            tokens_output = self.count_tokens(content, model)
            
            logger.info(
                "Local LLM completion generated",
                model=model,
                tokens_input=tokens_input,
                tokens_output=tokens_output,
                latency=latency
            )
            
            return {
                "content": content,
                "tokens_input": tokens_input,
                "tokens_output": tokens_output,
                "tokens_total": tokens_input + tokens_output,
                "cost_usd": 0.0,  # Local LLM has no cost
                "model": model,
                "provider": self.provider_name
            }
            
        except Exception as e:
            logger.error("Local LLM completion failed", error=str(e), model=model)
            raise
    
    async def generate_embedding(self, text: str, model: Optional[str] = None) -> List[float]:
        """Generate embedding using local model"""
        # For local embeddings, use sentence-transformers
        from sentence_transformers import SentenceTransformer
        
        model_name = settings.embedding_model
        model_obj = SentenceTransformer(model_name)
        embedding = model_obj.encode(text).tolist()
        
        logger.info("Local embedding generated", model=model_name)
        
        return embedding


class MockProvider(AIProvider):
    """Mock provider for testing"""
    
    def __init__(self):
        super().__init__("mock")
    
    async def generate_completion(
        self,
        prompt: str,
        max_tokens: int = 1000,
        temperature: float = 0.7,
        **kwargs
    ) -> Dict[str, Any]:
        """Generate mock completion"""
        content = f"Mock response for: {prompt[:50]}..."
        
        return {
            "content": content,
            "tokens_input": 100,
            "tokens_output": 50,
            "tokens_total": 150,
            "cost_usd": 0.0,
            "model": "mock-model",
            "provider": self.provider_name
        }
    
    async def generate_embedding(self, text: str) -> List[float]:
        """Generate mock embedding"""
        # Return a mock embedding vector
        return [0.1] * settings.embedding_dimension


def get_ai_provider(provider_name: Optional[str] = None) -> AIProvider:
    """
    Get AI provider instance
    
    Args:
        provider_name: Provider to use (openai, anthropic, local, mock)
    
    Returns:
        AIProvider instance
    """
    if settings.mock_providers or settings.test_mode:
        return MockProvider()
    
    provider_name = provider_name or "openai"
    
    if provider_name == "openai":
        return OpenAIProvider()
    elif provider_name == "anthropic":
        return AnthropicProvider()
    elif provider_name == "local":
        if not settings.local_llm_enabled:
            logger.warning("Local LLM not enabled, falling back to OpenAI")
            return OpenAIProvider()
        return LocalLLMProvider()
    elif provider_name == "mock":
        return MockProvider()
    else:
        raise ValueError(f"Unknown provider: {provider_name}")