"""
AI Jobs router - Main endpoint for AI task requests
"""
from fastapi import APIRouter, Depends, HTTPException, status, Header
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select
from typing import Optional, List
from datetime import datetime

from app.database import get_db
from app.models import AIJob
from app.schemas import (
    AIJobRequest, AIJobResponse, JobStatusResponse,
    JobStatus, ErrorResponse
)
from app.utils.observability import logger, metrics_collector
from app.utils.security import pii_redactor
from config import settings

router = APIRouter()


async def verify_api_key(x_api_key: Optional[str] = Header(None)):
    """Verify API key from Laravel"""
    if not x_api_key or x_api_key != settings.laravel_api_key:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or missing API key"
        )
    return x_api_key


@router.post("/jobs", response_model=AIJobResponse)
async def create_ai_job(
    request: AIJobRequest,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Create a new AI job
    
    Accepts job request, validates, stores in DB, enqueues to Celery,
    and returns job ID immediately.
    
    Supports idempotency via idempotency_key.
    """
    try:
        # Check for idempotency
        if request.idempotency_key:
            stmt = select(AIJob).where(
                AIJob.idempotency_key == request.idempotency_key
            )
            result = await db.execute(stmt)
            existing_job = result.scalar_one_or_none()
            
            if existing_job:
                logger.info(
                    "Returning existing job for idempotency key",
                    job_id=existing_job.id,
                    idempotency_key=request.idempotency_key
                )
                return AIJobResponse(
                    ai_job_id=existing_job.id,
                    status=JobStatus(existing_job.status),
                    estimated_time_seconds=None,
                    message="Job already exists (idempotent)"
                )
        
        # Redact PII from payload before storing
        redacted_payload = pii_redactor.redact_dict(request.payload)
        
        # Create job record
        job = AIJob(
            job_type=request.job_type.value,
            tenant_id=request.tenant_id,
            owner_id=request.owner_id,
            payload=redacted_payload,
            callback_url=request.callback_url,
            idempotency_key=request.idempotency_key,
            status="pending"
        )
        
        db.add(job)
        await db.commit()
        await db.refresh(job)
        
        logger.info(
            "AI job created",
            job_id=job.id,
            job_type=job.job_type,
            tenant_id=job.tenant_id
        )
        
        # Track metrics
        metrics_collector.track_job_request(job.job_type, job.tenant_id)
        
        # Enqueue to Celery
        from app.celery_app import process_ai_job
        task = process_ai_job.apply_async(
            args=[job.id],
            priority=request.priority
        )
        
        logger.info(
            "Job enqueued to Celery",
            job_id=job.id,
            celery_task_id=task.id
        )
        
        # Estimate completion time based on job type
        estimated_times = {
            "proposal": 45,
            "content_ideas": 30,
            "auto_reply": 15,
            "ads_generate": 60,
            "forecast": 120,
            "matchmaking": 90,
            "translate": 20,
            "contract_draft": 60,
            "support_rag": 10,
            "custom": 60
        }
        
        return AIJobResponse(
            ai_job_id=job.id,
            status=JobStatus.PENDING,
            estimated_time_seconds=estimated_times.get(request.job_type.value, 60),
            message="Job queued successfully"
        )
        
    except Exception as e:
        logger.error(
            "Failed to create AI job",
            error=str(e),
            job_type=request.job_type.value
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to create job: {str(e)}"
        )


@router.get("/jobs/{job_id}", response_model=JobStatusResponse)
async def get_job_status(
    job_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Get status and result of an AI job
    
    Returns job details including status, result (if completed),
    error message (if failed), and cost information.
    """
    try:
        stmt = select(AIJob).where(AIJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Job {job_id} not found"
            )
        
        return JobStatusResponse(
            ai_job_id=job.id,
            job_type=job.job_type,
            status=JobStatus(job.status),
            result=job.result,
            error_message=job.error_message,
            provider=job.provider,
            model_used=job.model_used,
            tokens_input=job.tokens_input,
            tokens_output=job.tokens_output,
            cost_usd=job.cost_usd,
            created_at=job.created_at,
            started_at=job.started_at,
            completed_at=job.completed_at,
            metadata=job.metadata
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to get job status", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to get job status: {str(e)}"
        )


@router.get("/jobs", response_model=List[JobStatusResponse])
async def list_jobs(
    tenant_id: Optional[str] = None,
    owner_id: Optional[str] = None,
    status_filter: Optional[str] = None,
    limit: int = 50,
    offset: int = 0,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    List AI jobs with optional filters
    
    Supports filtering by tenant_id, owner_id, and status.
    Returns paginated results.
    """
    try:
        stmt = select(AIJob)
        
        if tenant_id:
            stmt = stmt.where(AIJob.tenant_id == tenant_id)
        if owner_id:
            stmt = stmt.where(AIJob.owner_id == owner_id)
        if status_filter:
            stmt = stmt.where(AIJob.status == status_filter)
        
        stmt = stmt.order_by(AIJob.created_at.desc())
        stmt = stmt.limit(limit).offset(offset)
        
        result = await db.execute(stmt)
        jobs = result.scalars().all()
        
        return [
            JobStatusResponse(
                ai_job_id=job.id,
                job_type=job.job_type,
                status=JobStatus(job.status),
                result=job.result,
                error_message=job.error_message,
                provider=job.provider,
                model_used=job.model_used,
                tokens_input=job.tokens_input,
                tokens_output=job.tokens_output,
                cost_usd=job.cost_usd,
                created_at=job.created_at,
                started_at=job.started_at,
                completed_at=job.completed_at,
                metadata=job.metadata
            )
            for job in jobs
        ]
        
    except Exception as e:
        logger.error("Failed to list jobs", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to list jobs: {str(e)}"
        )


@router.delete("/jobs/{job_id}")
async def cancel_job(
    job_id: str,
    db: AsyncSession = Depends(get_db),
    api_key: str = Depends(verify_api_key)
):
    """
    Cancel a pending or processing job
    
    Only jobs in 'pending' or 'processing' status can be cancelled.
    """
    try:
        stmt = select(AIJob).where(AIJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Job {job_id} not found"
            )
        
        if job.status not in ["pending", "processing"]:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"Cannot cancel job in status: {job.status}"
            )
        
        job.status = "cancelled"
        job.completed_at = datetime.utcnow()
        await db.commit()
        
        logger.info("Job cancelled", job_id=job_id)
        
        return {"message": "Job cancelled successfully", "job_id": job_id}
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error("Failed to cancel job", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to cancel job: {str(e)}"
        )