#!/bin/bash

################################################################################
# AI Business Market - Automated VPS Deployment Script
# 
# This script will automatically setup and deploy your Laravel + Python project
# on a fresh Ubuntu 22.04 VPS server
#
# Usage:
#   1. Upload this script to your VPS: scp deploy.sh root@your-server-ip:/root/
#   2. SSH into VPS: ssh root@your-server-ip
#   3. Make executable: chmod +x deploy.sh
#   4. Run: ./deploy.sh
#
# Requirements:
#   - Fresh Ubuntu 22.04 VPS
#   - Root access
#   - At least 4GB RAM
#
################################################################################

set -e  # Exit on error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
PROJECT_NAME="ai-business-market"
DOMAIN=""  # Will be asked during setup
LARAVEL_DIR="/var/www/laravel"
PYTHON_DIR="/var/www/python"
DB_NAME="ai_business_db"
DB_USER="aiuser"
DB_PASSWORD=""  # Will be generated

# Functions
print_header() {
    echo -e "\n${BLUE}========================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}========================================${NC}\n"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_info() {
    echo -e "${YELLOW}ℹ $1${NC}"
}

generate_password() {
    openssl rand -base64 32 | tr -d "=+/" | cut -c1-25
}

# Start deployment
clear
print_header "AI Business Market - VPS Deployment"

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    print_error "Please run as root (use sudo)"
    exit 1
fi

# Get domain name
print_info "Enter your domain name (e.g., example.com):"
read -r DOMAIN

if [ -z "$DOMAIN" ]; then
    print_error "Domain name is required!"
    exit 1
fi

print_success "Domain set to: $DOMAIN"

# Generate passwords
DB_PASSWORD=$(generate_password)
PYTHON_API_KEY=$(openssl rand -hex 32)
CALLBACK_SECRET=$(openssl rand -hex 32)
SECRET_KEY=$(openssl rand -hex 32)
ENCRYPTION_KEY=$(openssl rand -hex 16)
JWT_SECRET=$(openssl rand -hex 32)

print_success "Generated secure passwords and keys"

# Save credentials
cat > /root/credentials.txt << EOF
===========================================
AI Business Market - Deployment Credentials
===========================================

Domain: $DOMAIN
Deployment Date: $(date)

Database:
---------
Database Name: $DB_NAME
Database User: $DB_USER
Database Password: $DB_PASSWORD

Integration Keys:
-----------------
Python API Key: $PYTHON_API_KEY
Callback Secret: $CALLBACK_SECRET
Secret Key: $SECRET_KEY
Encryption Key: $ENCRYPTION_KEY
JWT Secret: $JWT_SECRET

IMPORTANT: Keep this file secure!
Location: /root/credentials.txt

===========================================
EOF

print_success "Credentials saved to /root/credentials.txt"

################################################################################
# STEP 1: System Update
################################################################################
print_header "Step 1: Updating System"

apt update
apt upgrade -y
apt install -y curl wget git unzip software-properties-common

print_success "System updated"

################################################################################
# STEP 2: Install Nginx
################################################################################
print_header "Step 2: Installing Nginx"

apt install -y nginx
systemctl enable nginx
systemctl start nginx

print_success "Nginx installed"

################################################################################
# STEP 3: Install PHP 8.2
################################################################################
print_header "Step 3: Installing PHP 8.2"

add-apt-repository ppa:ondrej/php -y
apt update
apt install -y php8.2-fpm php8.2-cli php8.2-mysql php8.2-pgsql \
    php8.2-xml php8.2-curl php8.2-zip php8.2-gd php8.2-mbstring \
    php8.2-bcmath php8.2-redis php8.2-intl

print_success "PHP 8.2 installed"

################################################################################
# STEP 4: Install Composer
################################################################################
print_header "Step 4: Installing Composer"

curl -sS https://getcomposer.org/installer | php
mv composer.phar /usr/local/bin/composer
chmod +x /usr/local/bin/composer

print_success "Composer installed"

################################################################################
# STEP 5: Install Node.js 18
################################################################################
print_header "Step 5: Installing Node.js 18"

curl -fsSL https://deb.nodesource.com/setup_18.x | bash -
apt install -y nodejs

print_success "Node.js installed"

################################################################################
# STEP 6: Install Python 3.11
################################################################################
print_header "Step 6: Installing Python 3.11"

apt install -y python3.11 python3.11-venv python3-pip

print_success "Python 3.11 installed"

################################################################################
# STEP 7: Install PostgreSQL 15
################################################################################
print_header "Step 7: Installing PostgreSQL 15"

apt install -y postgresql-15 postgresql-contrib-15

# Configure PostgreSQL
sudo -u postgres psql << EOF
CREATE DATABASE $DB_NAME;
CREATE USER $DB_USER WITH ENCRYPTED PASSWORD '$DB_PASSWORD';
GRANT ALL PRIVILEGES ON DATABASE $DB_NAME TO $DB_USER;
ALTER DATABASE $DB_NAME OWNER TO $DB_USER;
\q
EOF

print_success "PostgreSQL installed and configured"

################################################################################
# STEP 8: Install Redis
################################################################################
print_header "Step 8: Installing Redis"

apt install -y redis-server
systemctl enable redis-server
systemctl start redis-server

print_success "Redis installed"

################################################################################
# STEP 9: Install Supervisor
################################################################################
print_header "Step 9: Installing Supervisor"

apt install -y supervisor
systemctl enable supervisor
systemctl start supervisor

print_success "Supervisor installed"

################################################################################
# STEP 10: Install Certbot (SSL)
################################################################################
print_header "Step 10: Installing Certbot"

apt install -y certbot python3-certbot-nginx

print_success "Certbot installed"

################################################################################
# STEP 11: Create Project Directories
################################################################################
print_header "Step 11: Creating Project Directories"

mkdir -p $LARAVEL_DIR
mkdir -p $PYTHON_DIR
chown -R www-data:www-data $LARAVEL_DIR
chown -R www-data:www-data $PYTHON_DIR

print_success "Project directories created"

################################################################################
# STEP 12: Configure Nginx for Laravel
################################################################################
print_header "Step 12: Configuring Nginx"

cat > /etc/nginx/sites-available/laravel << EOF
server {
    listen 80;
    server_name $DOMAIN www.$DOMAIN;
    root $LARAVEL_DIR/public;

    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";

    index index.php;

    charset utf-8;

    location / {
        try_files \$uri \$uri/ /index.php?\$query_string;
    }

    location = /favicon.ico { access_log off; log_not_found off; }
    location = /robots.txt  { access_log off; log_not_found off; }

    error_page 404 /index.php;

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME \$realpath_root\$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}

server {
    listen 80;
    server_name api.$DOMAIN;

    location / {
        proxy_pass http://localhost:8000;
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }
}
EOF

ln -sf /etc/nginx/sites-available/laravel /etc/nginx/sites-enabled/
rm -f /etc/nginx/sites-enabled/default
nginx -t
systemctl reload nginx

print_success "Nginx configured"

################################################################################
# STEP 13: Configure Supervisor for Python
################################################################################
print_header "Step 13: Configuring Supervisor"

cat > /etc/supervisor/conf.d/python-api.conf << EOF
[program:python-api]
command=$PYTHON_DIR/venv/bin/uvicorn app.main:app --host 0.0.0.0 --port 8000
directory=$PYTHON_DIR
user=www-data
autostart=true
autorestart=true
redirect_stderr=true
stdout_logfile=/var/log/python-api.log

[program:celery-worker]
command=$PYTHON_DIR/venv/bin/celery -A app.celery_app worker --loglevel=info
directory=$PYTHON_DIR
user=www-data
autostart=true
autorestart=true
redirect_stderr=true
stdout_logfile=/var/log/celery-worker.log

[program:celery-beat]
command=$PYTHON_DIR/venv/bin/celery -A app.celery_app beat --loglevel=info
directory=$PYTHON_DIR
user=www-data
autostart=true
autorestart=true
redirect_stderr=true
stdout_logfile=/var/log/celery-beat.log
EOF

print_success "Supervisor configured"

################################################################################
# STEP 14: Setup Firewall
################################################################################
print_header "Step 14: Configuring Firewall"

apt install -y ufw
ufw --force enable
ufw allow 22/tcp
ufw allow 80/tcp
ufw allow 443/tcp
ufw status

print_success "Firewall configured"

################################################################################
# STEP 15: Create Deployment Instructions
################################################################################
print_header "Step 15: Creating Deployment Instructions"

cat > /root/deploy_project.sh << 'DEPLOY_SCRIPT'
#!/bin/bash

# This script should be run after uploading your project files

set -e

LARAVEL_DIR="/var/www/laravel"
PYTHON_DIR="/var/www/python"

echo "========================================="
echo "Deploying Laravel Application"
echo "========================================="

cd $LARAVEL_DIR

# Install Laravel dependencies
composer install --no-dev --optimize-autoloader
npm install
npm run build

# Set permissions
chown -R www-data:www-data $LARAVEL_DIR
chmod -R 755 $LARAVEL_DIR
chmod -R 775 $LARAVEL_DIR/storage
chmod -R 775 $LARAVEL_DIR/bootstrap/cache

# Generate app key if not set
php artisan key:generate --force

# Run migrations
php artisan migrate --force

# Cache configuration
php artisan config:cache
php artisan route:cache
php artisan view:cache

echo "✓ Laravel deployed successfully"

echo "========================================="
echo "Deploying Python Application"
echo "========================================="

cd $PYTHON_DIR

# Create virtual environment
python3.11 -m venv venv
source venv/bin/activate

# Install dependencies
pip install --upgrade pip
pip install -r requirements.txt

# Set permissions
chown -R www-data:www-data $PYTHON_DIR

echo "✓ Python deployed successfully"

# Restart services
echo "========================================="
echo "Restarting Services"
echo "========================================="

systemctl restart php8.2-fpm
systemctl restart nginx
supervisorctl reread
supervisorctl update
supervisorctl restart all

echo "✓ All services restarted"

echo "========================================="
echo "Deployment Complete!"
echo "========================================="
echo ""
echo "Next steps:"
echo "1. Update .env files with your API keys"
echo "2. Run: certbot --nginx -d yourdomain.com -d www.yourdomain.com -d api.yourdomain.com"
echo "3. Test your application"
echo ""
DEPLOY_SCRIPT

chmod +x /root/deploy_project.sh

print_success "Deployment script created"

################################################################################
# STEP 16: Create Upload Instructions
################################################################################

cat > /root/UPLOAD_INSTRUCTIONS.txt << EOF
===========================================
How to Upload Your Project
===========================================

1. FROM YOUR LOCAL MACHINE, upload Laravel project:
   
   cd "c:/xampp/htdocs/AI business market"
   
   # Create Laravel .env file with production settings
   cp Laravel/.env.example Laravel/.env.production
   
   # Edit Laravel/.env.production and set:
   APP_ENV=production
   APP_DEBUG=false
   APP_URL=https://$DOMAIN
   DB_CONNECTION=sqlite
   PYTHON_API_URL=https://api.$DOMAIN
   PYTHON_API_KEY=$PYTHON_API_KEY
   PYTHON_CALLBACK_SECRET=$CALLBACK_SECRET
   
   # Upload Laravel
   rsync -avz --exclude 'node_modules' --exclude 'vendor' --exclude '.git' \\
     Laravel/ root@$DOMAIN:$LARAVEL_DIR/
   
   # Copy production .env
   scp Laravel/.env.production root@$DOMAIN:$LARAVEL_DIR/.env

2. Upload Python project:
   
   # Create Python .env file with production settings
   cp pytho/.env.example pytho/.env.production
   
   # Edit pytho/.env.production and set:
   APP_ENV=production
   APP_DEBUG=false
   DATABASE_URL=postgresql+asyncpg://$DB_USER:$DB_PASSWORD@localhost:5432/$DB_NAME
   REDIS_URL=redis://localhost:6379/0
   LARAVEL_BASE_URL=https://$DOMAIN
   LARAVEL_API_KEY=$PYTHON_API_KEY
   LARAVEL_CALLBACK_SECRET=$CALLBACK_SECRET
   SECRET_KEY=$SECRET_KEY
   ENCRYPTION_KEY=$ENCRYPTION_KEY
   JWT_SECRET=$JWT_SECRET
   OPENAI_API_KEY=your-openai-key-here
   
   # Upload Python
   rsync -avz --exclude 'venv' --exclude '__pycache__' --exclude '.git' \\
     pytho/ root@$DOMAIN:$PYTHON_DIR/
   
   # Copy production .env
   scp pytho/.env.production root@$DOMAIN:$PYTHON_DIR/.env

3. ON THE SERVER, run deployment:
   
   ssh root@$DOMAIN
   cd /root
   ./deploy_project.sh

4. Setup SSL certificate:
   
   certbot --nginx -d $DOMAIN -d www.$DOMAIN -d api.$DOMAIN

5. Test your application:
   
   https://$DOMAIN
   https://api.$DOMAIN/docs

===========================================
Important Files on Server:
===========================================

Credentials: /root/credentials.txt
Upload Instructions: /root/UPLOAD_INSTRUCTIONS.txt
Deploy Script: /root/deploy_project.sh

Laravel Directory: $LARAVEL_DIR
Python Directory: $PYTHON_DIR

Logs:
- Laravel: $LARAVEL_DIR/storage/logs/laravel.log
- Python API: /var/log/python-api.log
- Celery Worker: /var/log/celery-worker.log
- Nginx: /var/log/nginx/error.log

===========================================
EOF

print_success "Upload instructions created"

################################################################################
# Final Summary
################################################################################

print_header "Deployment Complete!"

echo -e "${GREEN}Server is ready for your project!${NC}\n"

echo -e "${YELLOW}Important Files Created:${NC}"
echo "  📄 /root/credentials.txt - Database & API credentials"
echo "  📄 /root/UPLOAD_INSTRUCTIONS.txt - How to upload your project"
echo "  📄 /root/deploy_project.sh - Deployment script"
echo ""

echo -e "${YELLOW}Next Steps:${NC}"
echo "  1. Read /root/UPLOAD_INSTRUCTIONS.txt"
echo "  2. Upload your Laravel project to $LARAVEL_DIR"
echo "  3. Upload your Python project to $PYTHON_DIR"
echo "  4. Run: ./deploy_project.sh"
echo "  5. Setup SSL: certbot --nginx -d $DOMAIN -d www.$DOMAIN -d api.$DOMAIN"
echo ""

echo -e "${YELLOW}Server Information:${NC}"
echo "  🌐 Domain: $DOMAIN"
echo "  🌐 API Domain: api.$DOMAIN"
echo "  📁 Laravel Directory: $LARAVEL_DIR"
echo "  📁 Python Directory: $PYTHON_DIR"
echo "  🗄️  Database: $DB_NAME"
echo ""

echo -e "${GREEN}View credentials: cat /root/credentials.txt${NC}"
echo -e "${GREEN}View upload instructions: cat /root/UPLOAD_INSTRUCTIONS.txt${NC}"
echo ""

print_success "Setup Complete! 🎉"