<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Services\PythonApiService;
use Illuminate\Support\Facades\Http;
use Illuminate\Foundation\Testing\RefreshDatabase;

class PythonIntegrationTest extends TestCase
{
    protected $pythonService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->pythonService = new PythonApiService();
    }

    /**
     * Test Python service health check
     *
     * @return void
     */
    public function test_python_service_health_check()
    {
        // Mock the HTTP response
        Http::fake([
            '*/health/readiness' => Http::response([
                'status' => 'healthy',
                'database' => 'connected',
                'redis' => 'connected',
            ], 200),
        ]);

        $isHealthy = $this->pythonService->checkHealth();

        $this->assertTrue($isHealthy);
    }

    /**
     * Test creating an AI job
     *
     * @return void
     */
    public function test_create_ai_job()
    {
        Http::fake([
            '*/v1/ai/jobs' => Http::response([
                'ai_job_id' => 'job-123',
                'status' => 'pending',
                'estimated_time_seconds' => 30,
                'message' => 'Job queued successfully',
            ], 200),
        ]);

        $result = $this->pythonService->createAiJob(
            jobType: 'content_ideas',
            tenantId: 'tenant-123',
            ownerId: 'user-456',
            payload: [
                'topic' => 'AI in marketing',
                'count' => 5,
            ]
        );

        $this->assertArrayHasKey('ai_job_id', $result);
        $this->assertEquals('pending', $result['status']);
        $this->assertEquals('job-123', $result['ai_job_id']);
    }

    /**
     * Test getting AI job status
     *
     * @return void
     */
    public function test_get_ai_job_status()
    {
        Http::fake([
            '*/v1/ai/jobs/*' => Http::response([
                'ai_job_id' => 'job-123',
                'status' => 'completed',
                'result' => [
                    'ideas' => [
                        'Idea 1',
                        'Idea 2',
                        'Idea 3',
                    ],
                ],
            ], 200),
        ]);

        $result = $this->pythonService->getAiJobStatus('job-123');

        $this->assertEquals('completed', $result['status']);
        $this->assertArrayHasKey('result', $result);
    }

    /**
     * Test publishing to social media
     *
     * @return void
     */
    public function test_publish_to_social_media()
    {
        Http::fake([
            '*/v1/publish' => Http::response([
                'publish_job_id' => 'pub-123',
                'status' => 'pending',
                'message' => 'Publishing job queued',
            ], 200),
        ]);

        $result = $this->pythonService->publishToSocial(
            platform: 'twitter',
            tenantId: 'tenant-123',
            ownerId: 'user-456',
            content: [
                'text' => 'Test post from Laravel',
            ]
        );

        $this->assertArrayHasKey('publish_job_id', $result);
        $this->assertEquals('pending', $result['status']);
    }

    /**
     * Test RAG query
     *
     * @return void
     */
    public function test_rag_query()
    {
        Http::fake([
            '*/v1/rag/query' => Http::response([
                'answer' => 'This is the answer from RAG',
                'sources' => [
                    ['id' => 'doc-1', 'title' => 'Document 1'],
                ],
                'confidence' => 0.95,
            ], 200),
        ]);

        $result = $this->pythonService->queryRag(
            tenantId: 'tenant-123',
            query: 'What is AI?',
            topK: 5
        );

        $this->assertArrayHasKey('answer', $result);
        $this->assertArrayHasKey('sources', $result);
    }

    /**
     * Test callback signature verification
     *
     * @return void
     */
    public function test_callback_signature_verification()
    {
        $payload = json_encode(['job_id' => 'job-123', 'status' => 'completed']);
        $secret = config('python.callback.secret');
        $signature = hash_hmac('sha256', $payload, $secret);

        $isValid = $this->pythonService->verifyCallbackSignature($payload, $signature);

        $this->assertTrue($isValid);
    }

    /**
     * Test invalid callback signature
     *
     * @return void
     */
    public function test_invalid_callback_signature()
    {
        $payload = json_encode(['job_id' => 'job-123', 'status' => 'completed']);
        $invalidSignature = 'invalid-signature';

        $isValid = $this->pythonService->verifyCallbackSignature($payload, $invalidSignature);

        $this->assertFalse($isValid);
    }

    /**
     * Test API error handling
     *
     * @return void
     */
    public function test_api_error_handling()
    {
        Http::fake([
            '*/v1/ai/jobs' => Http::response([
                'error' => 'Invalid request',
                'detail' => 'Missing required field: job_type',
            ], 400),
        ]);

        $this->expectException(\Exception::class);

        $this->pythonService->createAiJob(
            jobType: '',
            tenantId: 'tenant-123',
            ownerId: 'user-456',
            payload: []
        );
    }

    /**
     * Test configuration values
     *
     * @return void
     */
    public function test_python_configuration()
    {
        $this->assertNotEmpty(config('python.api.base_url'));
        $this->assertNotEmpty(config('python.api.api_key'));
        $this->assertNotEmpty(config('python.callback.secret'));
        $this->assertTrue(is_array(config('python.endpoints')));
        $this->assertTrue(is_array(config('python.job_types')));
    }
}