<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\User;
use App\Models\MessageThread;
use App\Models\Message;
use App\Models\MessageThreadParticipant;
use App\Models\Proposal;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class ProposalChatFeatureTest extends TestCase
{
    use RefreshDatabase;
    
    protected $brandUser;
    protected $creatorUser;
    protected $proposal;
    
    protected function setUp(): void
    {
        parent::setUp();
        
        // Create users
        $this->brandUser = User::factory()->create();
        $this->brandUser->assignRole('brand_admin');
        
        $this->creatorUser = User::factory()->create();
        $this->creatorUser->assignRole('creator');
        
        // Create a proposal
        $this->proposal = Proposal::factory()->create([
            'brand_id' => $this->brandUser->brand_id ?? 1,
            'creator_id' => $this->creatorUser->id,
            'title' => 'Test Proposal for Chat',
            'message' => 'This is a test proposal',
            'type' => 'Collaboration',
            'status' => 'sent',
        ]);
    }
    
    /** @test */
    public function a_thread_can_be_created_for_a_proposal()
    {
        Mail::fake();
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('threads.create'), [
                'recipient_id' => $this->creatorUser->id,
                'body' => 'I want to discuss this proposal with you',
                'proposal_id' => $this->proposal->id,
            ]);
            
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('message_threads', [
            'proposal_id' => $this->proposal->id,
        ]);
        
        // Verify the thread was created with the correct subject
        $thread = MessageThread::where('proposal_id', $this->proposal->id)->first();
        $this->assertEquals('Proposal Discussion', $thread->subject);
        
        $this->assertDatabaseHas('messages', [
            'sender_id' => $this->brandUser->id,
            'recipient_id' => $this->creatorUser->id,
            'message' => 'I want to discuss this proposal with you',
            'thread_id' => $thread->id,
        ]);
        
        // Verify notification was sent
        Mail::assertSent(\App\Mail\MessageReceived::class);
    }
    
    /** @test */
    public function existing_thread_is_reused_for_same_proposal()
    {
        Mail::fake();
        
        // Create first thread for proposal
        $response1 = $this->actingAs($this->brandUser)
            ->post(route('threads.create'), [
                'recipient_id' => $this->creatorUser->id,
                'body' => 'First message',
                'proposal_id' => $this->proposal->id,
            ]);
            
        $response1->assertStatus(200);
        $response1->assertJson(['success' => true]);
        
        // Get the thread ID
        $threadId = $response1->json('thread_id');
        
        // Create second thread for same proposal
        $response2 = $this->actingAs($this->creatorUser)
            ->post(route('threads.create'), [
                'recipient_id' => $this->brandUser->id,
                'body' => 'Second message',
                'proposal_id' => $this->proposal->id,
            ]);
            
        $response2->assertStatus(200);
        $response2->assertJson(['success' => true]);
        
        // Should reuse the same thread
        $this->assertEquals($threadId, $response2->json('thread_id'));
        
        // Should have 2 messages in the thread
        $this->assertEquals(2, Message::where('thread_id', $threadId)->count());
    }
    
    /** @test */
    public function proposal_detail_page_includes_link_to_chat_thread()
    {
        // Create a thread for the proposal
        $thread = MessageThread::create([
            'subject' => 'Proposal Discussion',
            'proposal_id' => $this->proposal->id,
        ]);
        
        // Add participants
        MessageThreadParticipant::create([
            'thread_id' => $thread->id,
            'user_id' => $this->brandUser->id,
            'joined_at' => now(),
        ]);
        
        MessageThreadParticipant::create([
            'thread_id' => $thread->id,
            'user_id' => $this->creatorUser->id,
            'joined_at' => now(),
        ]);
        
        // Brand user can see link to thread
        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.proposals.show', $this->proposal));
            
        $response->assertStatus(200);
        $response->assertSee(route('messages.show', $thread));
        
        // Creator user can see link to thread
        $response = $this->actingAs($this->creatorUser)
            ->get(route('creator.proposals.show', $this->proposal));
            
        $response->assertStatus(200);
        $response->assertSee(route('messages.show', $thread));
    }
    
    /** @test */
    public function messages_in_proposal_thread_are_properly_linked()
    {
        // Create a thread for the proposal
        $thread = MessageThread::create([
            'subject' => 'Proposal Discussion',
            'proposal_id' => $this->proposal->id,
        ]);
        
        // Add participants
        MessageThreadParticipant::create([
            'thread_id' => $thread->id,
            'user_id' => $this->brandUser->id,
            'joined_at' => now(),
        ]);
        
        MessageThreadParticipant::create([
            'thread_id' => $thread->id,
            'user_id' => $this->creatorUser->id,
            'joined_at' => now(),
        ]);
        
        // Create a message in the thread
        $message = Message::create([
            'thread_id' => $thread->id,
            'sender_id' => $this->brandUser->id,
            'recipient_id' => $this->creatorUser->id,
            'message' => 'Let\'s discuss the proposal details',
        ]);
        
        // Verify the relationships
        $this->assertEquals($this->proposal->id, $thread->proposal->id);
        $this->assertEquals($thread->id, $message->thread->id);
        $this->assertEquals($this->brandUser->id, $message->sender->id);
        $this->assertEquals($this->creatorUser->id, $message->recipient->id);
    }
}