<?php

namespace Tests\Feature;

use App\Models\Brand;
use App\Models\Campaign;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CampaignFeatureTest extends TestCase
{
    use RefreshDatabase;

    protected $brandUser;
    protected $creatorUser;
    protected $brand;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a brand
        $this->brand = Brand::factory()->create();

        // Create a brand admin user
        $this->brandUser = User::factory()->create([
            'brand_id' => $this->brand->id,
        ]);
        $this->brandUser->assignRole('brand_admin');

        // Create a creator user
        $this->creatorUser = User::factory()->create();
        $this->creatorUser->assignRole('creator');
    }

    /** @test */
    public function brand_admin_can_view_campaigns()
    {
        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.campaigns.index'));

        $response->assertStatus(200);
        $response->assertViewIs('brand.campaigns.index');
    }

    /** @test */
    public function brand_admin_can_create_campaign()
    {
        $response = $this->actingAs($this->brandUser)
            ->post(route('brand.campaigns.store'), [
                'name' => 'Test Campaign',
                'description' => 'This is a test campaign',
                'objective' => 'Increase brand awareness',
                'budget' => 5000,
                'currency' => 'USD',
                'start_date' => '2025-10-01',
                'end_date' => '2025-12-31',
                'status' => 'draft',
            ]);

        $response->assertStatus(302);
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('campaigns', [
            'name' => 'Test Campaign',
            'brand_id' => $this->brand->id,
        ]);
    }

    /** @test */
    public function brand_admin_can_view_campaign_details()
    {
        $campaign = Campaign::factory()->create([
            'brand_id' => $this->brand->id,
        ]);

        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.campaigns.show', $campaign));

        $response->assertStatus(200);
        $response->assertViewIs('brand.campaigns.show');
        $response->assertSee($campaign->name);
    }

    /** @test */
    public function brand_admin_can_update_campaign()
    {
        $campaign = Campaign::factory()->create([
            'brand_id' => $this->brand->id,
            'name' => 'Original Name',
        ]);

        $response = $this->actingAs($this->brandUser)
            ->put(route('brand.campaigns.update', $campaign), [
                'name' => 'Updated Name',
                'description' => 'Updated description',
                'objective' => 'Updated objective',
                'budget' => 6000,
                'currency' => 'USD',
                'start_date' => '2025-11-01',
                'end_date' => '2026-01-31',
                'status' => 'active',
            ]);

        $response->assertStatus(302);
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('campaigns', [
            'id' => $campaign->id,
            'name' => 'Updated Name',
            'status' => 'active',
        ]);
    }

    /** @test */
    public function creator_can_view_campaigns()
    {
        $response = $this->actingAs($this->creatorUser)
            ->get(route('creator.campaigns.index'));

        $response->assertStatus(200);
        $response->assertViewIs('creator.campaigns.index');
    }

    /** @test */
    public function creator_can_view_campaign_invitations()
    {
        $response = $this->actingAs($this->creatorUser)
            ->get(route('creator.campaigns.invitations.index'));

        $response->assertStatus(200);
        $response->assertViewIs('creator.campaigns.invitations');
    }

    /** @test */
    public function creator_can_view_campaign_details()
    {
        $campaign = Campaign::factory()->create([
            'brand_id' => $this->brand->id,
        ]);

        // Attach creator to campaign
        $campaign->creators()->attach($this->creatorUser->id, [
            'invited_by' => $this->brandUser->id,
            'status' => 'accepted',
        ]);

        $response = $this->actingAs($this->creatorUser)
            ->get(route('creator.campaigns.show', $campaign));

        $response->assertStatus(200);
        $response->assertViewIs('creator.campaigns.show');
        $response->assertSee($campaign->name);
    }
}