<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Proposal;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandProposalLifecycleTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }
    }

    public function test_brand_admin_can_access_proposals_index()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.proposals.index'));

        $response->assertStatus(200);
        $response->assertSee('Proposals');
    }

    public function test_brand_admin_can_create_proposal()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.proposals.create'));

        $response->assertStatus(200);
        $response->assertSee('Create Proposal');
    }

    public function test_brand_admin_can_view_proposal_details()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'sent',
        ]);

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.proposals.show', $proposal));

        $response->assertStatus(200);
        $response->assertSee('Proposal Details');
        $response->assertSee('Test Proposal');
    }

    public function test_creator_cannot_access_brand_proposals_pages()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.proposals.index'));

        $response->assertStatus(403);

        $response = $this->actingAs($creator)
            ->get(route('brand.proposals.create'));

        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_proposals()
    {
        $response = $this->get(route('brand.proposals.index'));
        $response->assertRedirect(route('login'));

        $response = $this->get(route('brand.proposals.create'));
        $response->assertRedirect(route('login'));
    }
}