<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\MessageThread;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandMessagesTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }
    }

    public function test_brand_admin_can_access_messages_index()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.messages.index'));

        $response->assertStatus(200);
        $response->assertSee('Messages');
    }

    public function test_brand_admin_can_view_message_thread()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $thread = MessageThread::factory()->create([
            'subject' => 'Test Message Thread',
        ]);

        // Add participants
        $thread->participants()->create([
            'user_id' => $brandAdmin->id,
        ]);

        $thread->participants()->create([
            'user_id' => $creator->id,
        ]);

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.messages.show', $thread));

        $response->assertStatus(200);
        $response->assertSee('Message Thread');
        $response->assertSee('Test Message Thread');
    }

    public function test_creator_cannot_access_brand_messages_pages()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.messages.index'));

        $response->assertStatus(403);

        $thread = MessageThread::factory()->create();
        $response = $this->actingAs($creator)
            ->get(route('brand.messages.show', $thread));

        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_messages()
    {
        $response = $this->get(route('brand.messages.index'));
        $response->assertRedirect(route('login'));

        $thread = MessageThread::factory()->create();
        $response = $this->get(route('brand.messages.show', $thread));
        $response->assertRedirect(route('login'));
    }
}