<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Brand;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandMarketplaceTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }
    }

    public function test_brand_admin_can_access_marketplace_index()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.marketplace.index'));

        $response->assertStatus(200);
        $response->assertSee('Marketplace');
    }

    public function test_brand_admin_can_access_shortlists()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.marketplace.shortlists.index'));

        $response->assertStatus(200);
        $response->assertSee('My Shortlists');
    }

    public function test_brand_admin_can_access_recommendations()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.marketplace.recommendations.index'));

        $response->assertStatus(200);
        $response->assertSee('Recommended Creators');
    }

    public function test_creator_cannot_access_marketplace_pages()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.marketplace.index'));

        $response->assertStatus(403);

        $response = $this->actingAs($creator)
            ->get(route('brand.marketplace.shortlists.index'));

        $response->assertStatus(403);

        $response = $this->actingAs($creator)
            ->get(route('brand.marketplace.recommendations.index'));

        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_marketplace()
    {
        $response = $this->get(route('brand.marketplace.index'));
        $response->assertRedirect(route('login'));

        $response = $this->get(route('brand.marketplace.shortlists.index'));
        $response->assertRedirect(route('login'));

        $response = $this->get(route('brand.marketplace.recommendations.index'));
        $response->assertRedirect(route('login'));
    }
}