<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandCreatorsFilterTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'password' => bcrypt('password123'),
        ]);
        
        $this->brandAdmin->assignRole('brand_admin');
    }

    public function test_creators_page_loads_with_unfiltered_list()
    {
        // Create some creators with different influencer types
        $creator1 = User::factory()->create([
            'name' => 'Tech Creator',
            'email' => 'tech@creator.com',
            'influencer_type' => 'Tech',
        ]);
        $creator1->assignRole('creator');
        
        $creator2 = User::factory()->create([
            'name' => 'Gaming Creator',
            'email' => 'gaming@creator.com',
            'influencer_type' => 'Gaming',
        ]);
        $creator2->assignRole('creator');
        
        $creator3 = User::factory()->create([
            'name' => 'Other Creator',
            'email' => 'other@creator.com',
            'influencer_type' => 'Other',
            'influencer_type_custom' => 'Photography',
        ]);
        $creator3->assignRole('creator');

        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.index'));

        $response->assertStatus(200);
        $response->assertSee('Creators');
        $response->assertSee('Tech Creator');
        $response->assertSee('Gaming Creator');
        $response->assertSee('Other Creator');
        $response->assertSee('Photography'); // Custom type should be displayed
        $response->assertSee('Tech');
        $response->assertSee('Gaming');
    }

    public function test_filter_by_single_influencer_type()
    {
        // Create creators with different influencer types
        $creator1 = User::factory()->create([
            'name' => 'Tech Creator',
            'email' => 'tech@creator.com',
            'influencer_type' => 'Tech',
        ]);
        $creator1->assignRole('creator');
        
        $creator2 = User::factory()->create([
            'name' => 'Gaming Creator',
            'email' => 'gaming@creator.com',
            'influencer_type' => 'Gaming',
        ]);
        $creator2->assignRole('creator');

        // Filter by Tech type
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.index', ['types' => ['Tech']]));

        $response->assertStatus(200);
        $response->assertSee('Tech Creator');
        $response->assertDontSee('Gaming Creator');
    }

    public function test_filter_by_multiple_influencer_types()
    {
        // Create creators with different influencer types
        $creator1 = User::factory()->create([
            'name' => 'Tech Creator',
            'email' => 'tech@creator.com',
            'influencer_type' => 'Tech',
        ]);
        $creator1->assignRole('creator');
        
        $creator2 = User::factory()->create([
            'name' => 'Gaming Creator',
            'email' => 'gaming@creator.com',
            'influencer_type' => 'Gaming',
        ]);
        $creator2->assignRole('creator');
        
        $creator3 = User::factory()->create([
            'name' => 'Beauty Creator',
            'email' => 'beauty@creator.com',
            'influencer_type' => 'Beauty',
        ]);
        $creator3->assignRole('creator');

        // Filter by Tech and Gaming types
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.index', ['types' => ['Tech', 'Gaming']]));

        $response->assertStatus(200);
        $response->assertSee('Tech Creator');
        $response->assertSee('Gaming Creator');
        $response->assertDontSee('Beauty Creator');
    }

    public function test_filter_by_other_influencer_type()
    {
        // Create creators
        $creator1 = User::factory()->create([
            'name' => 'Other Creator',
            'email' => 'other@creator.com',
            'influencer_type' => 'Other',
            'influencer_type_custom' => 'Photography',
        ]);
        $creator1->assignRole('creator');
        
        $creator2 = User::factory()->create([
            'name' => 'Tech Creator',
            'email' => 'tech@creator.com',
            'influencer_type' => 'Tech',
        ]);
        $creator2->assignRole('creator');

        // Filter by Other type
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.index', ['types' => ['Other']]));

        $response->assertStatus(200);
        $response->assertSee('Other Creator');
        $response->assertDontSee('Tech Creator');
    }

    public function test_filter_persists_in_pagination_links()
    {
        // Create many creators to trigger pagination
        for ($i = 1; $i <= 30; $i++) {
            $creator = User::factory()->create([
                'name' => "Creator {$i}",
                'email' => "creator{$i}@test.com",
                'influencer_type' => 'Tech',
            ]);
            $creator->assignRole('creator');
        }

        // Filter by Tech type and go to page 2
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.index', ['types' => ['Tech'], 'page' => 2]));

        $response->assertStatus(200);
        
        // Check that pagination links include the filter
        $response->assertSee('types%5B0%5D=Tech', false); // URL encoded 'types[]=Tech'
    }
}