@extends('layouts.app')

@section('content')
<div class="container mx-auto px-4 py-8">
    <h1 class="text-3xl font-bold text-gray-900 mb-8">Manage Subscription</h1>

    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-6">
            {{ session('success') }}
        </div>
    @endif

    @if(session('error'))
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
            {{ session('error') }}
        </div>
    @endif

    @if(session('info'))
        <div class="bg-blue-100 border border-blue-400 text-blue-700 px-4 py-3 rounded mb-6">
            {{ session('info') }}
        </div>
    @endif

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <!-- Current Plan -->
        <div class="lg:col-span-2">
            <div class="bg-white rounded-lg shadow-lg p-6 mb-6">
                <h2 class="text-2xl font-semibold mb-4">Current Plan</h2>
                
                <div class="flex items-center justify-between mb-6">
                    <div>
                        <h3 class="text-3xl font-bold text-blue-600">{{ $currentPlan->name }}</h3>
                        <p class="text-gray-600">{{ $currentPlan->price_text }}</p>
                    </div>
                    <div class="text-right">
                        <span class="inline-block px-3 py-1 rounded-full text-sm font-semibold
                            {{ $subscription->isActive() ? 'bg-green-100 text-green-800' : 
                               ($subscription->onTrial() ? 'bg-blue-100 text-blue-800' : 
                               ($subscription->isCancelled() ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-gray-800')) }}">
                            {{ ucfirst($subscription->status) }}
                        </span>
                    </div>
                </div>

                <!-- Subscription Details -->
                <div class="grid grid-cols-2 gap-4 mb-6">
                    @if($subscription->onTrial())
                        <div class="bg-blue-50 p-4 rounded-lg">
                            <p class="text-sm text-gray-600">Trial Ends</p>
                            <p class="text-lg font-semibold">{{ $subscription->trial_ends_at->format('M d, Y') }}</p>
                            <p class="text-xs text-gray-500">{{ $subscription->trialDaysRemaining() }} days remaining</p>
                        </div>
                    @endif
                    
                    <div class="bg-gray-50 p-4 rounded-lg">
                        <p class="text-sm text-gray-600">Next Billing Date</p>
                        <p class="text-lg font-semibold">{{ $subscription->current_period_end->format('M d, Y') }}</p>
                        <p class="text-xs text-gray-500">{{ $subscription->daysRemaining() }} days remaining</p>
                    </div>

                    @if($subscription->isCancelled())
                        <div class="bg-red-50 p-4 rounded-lg col-span-2">
                            <p class="text-sm text-gray-600">Subscription Ends</p>
                            <p class="text-lg font-semibold text-red-600">{{ $subscription->ends_at->format('M d, Y') }}</p>
                            <p class="text-xs text-gray-500">Access until end of billing period</p>
                        </div>
                    @endif
                </div>

                <!-- Actions -->
                <div class="flex gap-4">
                    @if(!$subscription->isCancelled())
                        <a href="{{ route('subscription.plans') }}" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg text-center transition duration-200">
                            Change Plan
                        </a>
                        
                        <form action="{{ route('subscription.cancel-subscription') }}" method="POST" class="flex-1" onsubmit="return confirm('Are you sure you want to cancel your subscription?');">
                            @csrf
                            <button type="submit" class="w-full bg-red-600 hover:bg-red-700 text-white font-semibold py-2 px-4 rounded-lg transition duration-200">
                                Cancel Subscription
                            </button>
                        </form>
                    @else
                        <form action="{{ route('subscription.resume-subscription') }}" method="POST" class="flex-1">
                            @csrf
                            <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-lg transition duration-200">
                                Resume Subscription
                            </button>
                        </form>
                    @endif
                </div>
            </div>

            <!-- Usage Limits -->
            <div class="bg-white rounded-lg shadow-lg p-6 mb-6">
                <h2 class="text-2xl font-semibold mb-4">Usage This Month</h2>
                
                <div class="space-y-4">
                    @foreach($limits as $featureName => $limit)
                        <div>
                            <div class="flex justify-between mb-1">
                                <span class="text-sm font-medium text-gray-700">{{ ucwords(str_replace('_', ' ', $featureName)) }}</span>
                                <span class="text-sm text-gray-600">
                                    {{ $limit['used'] }} / {{ $limit['unlimited'] ? '∞' : $limit['limit'] }}
                                </span>
                            </div>
                            @if(!$limit['unlimited'])
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-blue-600 h-2 rounded-full" style="width: {{ min(100, ($limit['used'] / max(1, $limit['limit'])) * 100) }}%"></div>
                                </div>
                            @else
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-green-600 h-2 rounded-full" style="width: 100%"></div>
                                </div>
                            @endif
                        </div>
                    @endforeach
                </div>

                @if(array_filter($limits, fn($l) => !$l['unlimited'] && $l['used'] >= $l['limit']))
                    <div class="mt-4 bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <p class="text-sm text-yellow-800">
                            <strong>Limit Reached:</strong> You've reached your monthly limit for some features. 
                            <a href="{{ route('subscription.plans') }}" class="text-blue-600 hover:text-blue-700 underline">Upgrade your plan</a> for more.
                        </p>
                    </div>
                @endif
            </div>

            <!-- Recent Invoices -->
            <div class="bg-white rounded-lg shadow-lg p-6">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-2xl font-semibold">Recent Invoices</h2>
                    <a href="{{ route('subscription.invoices') }}" class="text-blue-600 hover:text-blue-700 text-sm">View All</a>
                </div>

                @if($invoices->count() > 0)
                    <div class="space-y-3">
                        @foreach($invoices as $invoice)
                            <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                <div>
                                    <p class="font-semibold">{{ $invoice->invoice_number }}</p>
                                    <p class="text-sm text-gray-600">{{ $invoice->issued_at->format('M d, Y') }}</p>
                                </div>
                                <div class="text-right">
                                    <p class="font-semibold">{{ $invoice->formatted_total }}</p>
                                    <a href="{{ route('subscription.invoices.download', $invoice) }}" class="text-sm text-blue-600 hover:text-blue-700">
                                        Download PDF
                                    </a>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <p class="text-gray-500 text-center py-4">No invoices yet</p>
                @endif
            </div>
        </div>

        <!-- Sidebar -->
        <div class="lg:col-span-1">
            <!-- Plan Features -->
            <div class="bg-white rounded-lg shadow-lg p-6 mb-6">
                <h3 class="font-semibold text-lg mb-4">Your Plan Includes</h3>
                <ul class="space-y-2">
                    @foreach($currentPlan->features as $feature)
                        <li class="flex items-start text-sm">
                            <svg class="w-4 h-4 text-green-500 mr-2 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
                            </svg>
                            <span class="text-gray-700">{{ $feature }}</span>
                        </li>
                    @endforeach
                </ul>
            </div>

            <!-- Upgrade Prompt -->
            @if($currentPlan->slug !== 'enterprise')
                <div class="bg-gradient-to-r from-blue-500 to-purple-600 rounded-lg shadow-lg p-6 text-white">
                    <h3 class="font-semibold text-lg mb-2">Need More?</h3>
                    <p class="text-sm mb-4 opacity-90">Upgrade to unlock advanced features and higher limits</p>
                    <a href="{{ route('subscription.plans') }}" class="block w-full bg-white text-blue-600 font-semibold py-2 px-4 rounded-lg text-center hover:bg-gray-100 transition duration-200">
                        View Plans
                    </a>
                </div>
            @endif

            <!-- Support -->
            <div class="bg-white rounded-lg shadow-lg p-6 mt-6">
                <h3 class="font-semibold text-lg mb-4">Need Help?</h3>
                <p class="text-sm text-gray-600 mb-4">Our support team is here to help you with any questions.</p>
                <a href="mailto:support@example.com" class="block w-full bg-gray-200 hover:bg-gray-300 text-gray-800 font-semibold py-2 px-4 rounded-lg text-center transition duration-200">
                    Contact Support
                </a>
            </div>
        </div>
    </div>
</div>
@endsection