/**
 * AI Business Market - Analytics Tracking Helper
 * Centralized analytics event tracking for GA4, Segment, and custom events
 */

class AnalyticsTracker {
    constructor() {
        this.initialized = false;
        this.queue = [];
        this.debug = import.meta.env.DEV || false;
        
        // Initialize when DOM is ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            this.init();
        }
    }

    /**
     * Initialize analytics tracking
     */
    init() {
        if (this.initialized) return;
        
        this.log('Analytics initialized');
        this.initialized = true;
        
        // Process queued events
        this.queue.forEach(event => this.track(event.name, event.properties));
        this.queue = [];
        
        // Set up automatic tracking
        this.setupAutoTracking();
    }

    /**
     * Track a custom event
     * @param {string} eventName - Name of the event
     * @param {object} properties - Event properties
     */
    track(eventName, properties = {}) {
        if (!this.initialized) {
            this.queue.push({ name: eventName, properties });
            return;
        }

        // Add common properties
        const enrichedProperties = {
            ...properties,
            timestamp: new Date().toISOString(),
            page_url: window.location.href,
            page_title: document.title,
            user_agent: navigator.userAgent,
            screen_resolution: `${window.screen.width}x${window.screen.height}`,
            viewport_size: `${window.innerWidth}x${window.innerHeight}`
        };

        // Track with Google Analytics 4
        if (typeof gtag !== 'undefined') {
            gtag('event', eventName, enrichedProperties);
            this.log('GA4 Event:', eventName, enrichedProperties);
        }

        // Track with Segment (if available)
        if (typeof analytics !== 'undefined' && analytics.track) {
            analytics.track(eventName, enrichedProperties);
            this.log('Segment Event:', eventName, enrichedProperties);
        }

        // Track with Facebook Pixel (if available)
        if (typeof fbq !== 'undefined') {
            fbq('trackCustom', eventName, enrichedProperties);
            this.log('FB Pixel Event:', eventName, enrichedProperties);
        }

        // Custom event for internal tracking
        this.dispatchCustomEvent(eventName, enrichedProperties);
    }

    /**
     * Track page view
     * @param {string} pagePath - Page path
     * @param {string} pageTitle - Page title
     */
    pageView(pagePath = window.location.pathname, pageTitle = document.title) {
        this.track('page_view', {
            page_path: pagePath,
            page_title: pageTitle
        });
    }

    /**
     * Track pricing view
     * @param {string} planId - Plan ID
     * @param {string} role - User role
     */
    pricingView(planId, role = 'all') {
        this.track('pricing_view', {
            plan_id: planId,
            role: role
        });
    }

    /**
     * Track plan click
     * @param {string} planId - Plan ID
     * @param {string} planName - Plan name
     * @param {string} role - User role
     */
    planClick(planId, planName, role = 'all') {
        this.track('plan_click', {
            plan_id: planId,
            plan_name: planName,
            role: role
        });
    }

    /**
     * Track checkout initiated
     * @param {string} planId - Plan ID
     * @param {string} planName - Plan name
     * @param {number} amount - Amount
     * @param {string} userId - User ID (optional)
     */
    checkoutInitiated(planId, planName, amount, userId = null) {
        this.track('checkout_initiated', {
            plan_id: planId,
            plan_name: planName,
            amount: amount,
            user_id: userId,
            currency: 'INR'
        });
    }

    /**
     * Track payment success
     * @param {string} planId - Plan ID
     * @param {number} amount - Amount
     * @param {string} razorpayId - Razorpay payment ID
     * @param {string} userId - User ID
     */
    paymentSuccess(planId, amount, razorpayId, userId) {
        this.track('payment_success', {
            plan_id: planId,
            amount: amount,
            razorpay_payment_id: razorpayId,
            user_id: userId,
            currency: 'INR',
            transaction_id: razorpayId
        });

        // Also track as purchase for e-commerce
        if (typeof gtag !== 'undefined') {
            gtag('event', 'purchase', {
                transaction_id: razorpayId,
                value: amount,
                currency: 'INR',
                items: [{
                    item_id: planId,
                    item_name: `Subscription Plan ${planId}`,
                    price: amount,
                    quantity: 1
                }]
            });
        }
    }

    /**
     * Track trial started
     * @param {string} planId - Plan ID
     * @param {number} trialDays - Trial duration in days
     */
    trialStarted(planId, trialDays) {
        this.track('trial_started', {
            plan_id: planId,
            trial_days: trialDays
        });
    }

    /**
     * Track upgrade initiated
     * @param {string} fromPlan - Current plan ID
     * @param {string} toPlan - Target plan ID
     */
    upgradeInitiated(fromPlan, toPlan) {
        this.track('upgrade_initiated', {
            from_plan: fromPlan,
            to_plan: toPlan
        });
    }

    /**
     * Track contact sales
     * @param {string} source - Source of contact (e.g., 'pricing_page', 'enterprise_cta')
     */
    contactSales(source) {
        this.track('contact_sales', {
            source: source
        });
    }

    /**
     * Track form submission
     * @param {string} formName - Name of the form
     * @param {object} formData - Form data (sanitized)
     */
    formSubmit(formName, formData = {}) {
        this.track('form_submit', {
            form_name: formName,
            ...formData
        });
    }

    /**
     * Track CTA click
     * @param {string} ctaText - CTA button text
     * @param {string} ctaLocation - Location of CTA
     */
    ctaClick(ctaText, ctaLocation) {
        this.track('cta_click', {
            cta_text: ctaText,
            cta_location: ctaLocation
        });
    }

    /**
     * Track video play
     * @param {string} videoTitle - Video title
     * @param {string} videoUrl - Video URL
     */
    videoPlay(videoTitle, videoUrl) {
        this.track('video_play', {
            video_title: videoTitle,
            video_url: videoUrl
        });
    }

    /**
     * Track search
     * @param {string} searchTerm - Search query
     * @param {number} resultsCount - Number of results
     */
    search(searchTerm, resultsCount = 0) {
        this.track('search', {
            search_term: searchTerm,
            results_count: resultsCount
        });
    }

    /**
     * Track scroll depth
     * @param {number} percentage - Scroll percentage
     */
    scrollDepth(percentage) {
        this.track('scroll_depth', {
            percentage: percentage
        });
    }

    /**
     * Track time on page
     * @param {number} seconds - Time spent in seconds
     */
    timeOnPage(seconds) {
        this.track('time_on_page', {
            seconds: seconds,
            minutes: Math.floor(seconds / 60)
        });
    }

    /**
     * Set up automatic tracking for common interactions
     */
    setupAutoTracking() {
        // Track outbound links
        document.addEventListener('click', (e) => {
            const link = e.target.closest('a');
            if (link && link.hostname !== window.location.hostname) {
                this.track('outbound_link_click', {
                    url: link.href,
                    text: link.textContent.trim()
                });
            }
        });

        // Track scroll depth milestones
        let scrollMilestones = [25, 50, 75, 100];
        let trackedMilestones = new Set();
        
        window.addEventListener('scroll', () => {
            const scrollPercentage = Math.round(
                (window.scrollY / (document.documentElement.scrollHeight - window.innerHeight)) * 100
            );
            
            scrollMilestones.forEach(milestone => {
                if (scrollPercentage >= milestone && !trackedMilestones.has(milestone)) {
                    trackedMilestones.add(milestone);
                    this.scrollDepth(milestone);
                }
            });
        });

        // Track time on page
        let startTime = Date.now();
        window.addEventListener('beforeunload', () => {
            const timeSpent = Math.floor((Date.now() - startTime) / 1000);
            if (timeSpent > 5) { // Only track if more than 5 seconds
                this.timeOnPage(timeSpent);
            }
        });

        // Track visibility changes
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.track('page_hidden');
            } else {
                this.track('page_visible');
            }
        });
    }

    /**
     * Dispatch custom event for internal use
     * @param {string} eventName - Event name
     * @param {object} properties - Event properties
     */
    dispatchCustomEvent(eventName, properties) {
        const event = new CustomEvent('analytics:track', {
            detail: { eventName, properties }
        });
        window.dispatchEvent(event);
    }

    /**
     * Capture UTM parameters and store in session
     */
    captureUTM() {
        const urlParams = new URLSearchParams(window.location.search);
        const utmParams = {};
        
        ['utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content'].forEach(param => {
            if (urlParams.has(param)) {
                utmParams[param] = urlParams.get(param);
            }
        });

        if (Object.keys(utmParams).length > 0) {
            sessionStorage.setItem('utm_params', JSON.stringify(utmParams));
            this.track('utm_captured', utmParams);
        }
    }

    /**
     * Get stored UTM parameters
     * @returns {object} UTM parameters
     */
    getUTM() {
        const stored = sessionStorage.getItem('utm_params');
        return stored ? JSON.parse(stored) : {};
    }

    /**
     * Log debug messages
     * @param {...any} args - Arguments to log
     */
    log(...args) {
        if (this.debug) {
            console.log('[Analytics]', ...args);
        }
    }

    /**
     * Identify user (for authenticated sessions)
     * @param {string} userId - User ID
     * @param {object} traits - User traits
     */
    identify(userId, traits = {}) {
        if (typeof gtag !== 'undefined') {
            gtag('set', 'user_id', userId);
            gtag('set', 'user_properties', traits);
        }

        if (typeof analytics !== 'undefined' && analytics.identify) {
            analytics.identify(userId, traits);
        }

        this.log('User identified:', userId, traits);
    }
}

// Create global instance
window.tracker = new AnalyticsTracker();

// Capture UTM parameters on page load
window.tracker.captureUTM();

// Export for module usage
export default window.tracker;